import { JsonRpcError } from 'jsonrpc-lite';
import { AppObjectRegistry } from '../../AppObjectRegistry.ts';
import { MessageExtender } from '../../lib/accessors/extenders/MessageExtender.ts';
import { RoomExtender } from '../../lib/accessors/extenders/RoomExtender.ts';
import { MessageBuilder } from '../../lib/accessors/builders/MessageBuilder.ts';
import { RoomBuilder } from '../../lib/accessors/builders/RoomBuilder.ts';
import { AppAccessorsInstance } from '../../lib/accessors/mod.ts';
import { require } from '../../lib/require.ts';
import createRoom from '../../lib/roomFactory.ts';
import { Room } from '../../lib/room.ts';
const { AppsEngineException } = require('@rocket.chat/apps-engine/definition/exceptions/AppsEngineException.js');
export default async function handleListener(evtInterface, params) {
  const app = AppObjectRegistry.get('app');
  const eventExecutor = app?.[evtInterface];
  if (typeof eventExecutor !== 'function') {
    return JsonRpcError.methodNotFound({
      message: 'Invalid event interface called on app'
    });
  }
  if (!Array.isArray(params) || params.length < 1 || params.length > 2) {
    return JsonRpcError.invalidParams(null);
  }
  try {
    const args = parseArgs({
      AppAccessorsInstance
    }, evtInterface, params);
    return await eventExecutor.apply(app, args);
  } catch (e) {
    if (e instanceof JsonRpcError) {
      return e;
    }
    if (e instanceof AppsEngineException) {
      return new JsonRpcError(e.message, AppsEngineException.JSONRPC_ERROR_CODE, {
        name: e.name
      });
    }
    return JsonRpcError.internalError({
      message: e.message
    });
  }
}
export function parseArgs(deps, evtMethod, params) {
  const { AppAccessorsInstance } = deps;
  /**
	 * param1 is the context for the event handler execution
	 * param2 is an optional extra content that some hanlers require
	 */ const [param1, param2] = params;
  if (!param1) {
    throw JsonRpcError.invalidParams(null);
  }
  let context = param1;
  if (evtMethod.includes('Message')) {
    context = hydrateMessageObjects(context);
  } else if (evtMethod.endsWith('RoomUserJoined') || evtMethod.endsWith('RoomUserLeave')) {
    context.room = createRoom(context.room, AppAccessorsInstance.getSenderFn());
  } else if (evtMethod.includes('PreRoom')) {
    context = createRoom(context, AppAccessorsInstance.getSenderFn());
  }
  const args = [
    context,
    AppAccessorsInstance.getReader(),
    AppAccessorsInstance.getHttp()
  ];
  // "check" events will only go this far - (context, reader, http)
  if (evtMethod.startsWith('check')) {
    // "checkPostMessageDeleted" has an extra param - (context, reader, http, extraContext)
    if (param2) {
      args.push(hydrateMessageObjects(param2));
    }
    return args;
  }
  // From this point on, all events will require (reader, http, persistence) injected
  args.push(AppAccessorsInstance.getPersistence());
  // "extend" events have an additional "Extender" param - (context, extender, reader, http, persistence)
  if (evtMethod.endsWith('Extend')) {
    if (evtMethod.includes('Message')) {
      args.splice(1, 0, new MessageExtender(param1));
    } else if (evtMethod.includes('Room')) {
      args.splice(1, 0, new RoomExtender(param1));
    }
    return args;
  }
  // "Modify" events have an additional "Builder" param - (context, builder, reader, http, persistence)
  if (evtMethod.endsWith('Modify')) {
    if (evtMethod.includes('Message')) {
      args.splice(1, 0, new MessageBuilder(param1));
    } else if (evtMethod.includes('Room')) {
      args.splice(1, 0, new RoomBuilder(param1));
    }
    return args;
  }
  // From this point on, all events will require (reader, http, persistence, modifier) injected
  args.push(AppAccessorsInstance.getModifier());
  // This guy gets an extra one
  if (evtMethod === 'executePostMessageDeleted') {
    if (!param2) {
      throw JsonRpcError.invalidParams(null);
    }
    args.push(hydrateMessageObjects(param2));
  }
  return args;
}
/**
 * Hydrate the context object with the correct IMessage
 *
 * Some information is lost upon serializing the data from listeners through the pipes,
 * so here we hydrate the complete object as necessary
 */ function hydrateMessageObjects(context) {
  if (objectIsRawMessage(context)) {
    context.room = createRoom(context.room, AppAccessorsInstance.getSenderFn());
  } else if (context?.message) {
    context.message = hydrateMessageObjects(context.message);
  }
  return context;
}
function objectIsRawMessage(value) {
  if (!value) return false;
  const { id, room, sender, createdAt } = value;
  // Check if we have the fields of a message and the room hasn't already been hydrated
  return !!(id && room && sender && createdAt) && !(room instanceof Room);
}
//# sourceMappingURL=data:application/json;base64,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