import { JsonRpcError } from 'jsonrpc-lite';
import { AppObjectRegistry } from '../AppObjectRegistry.ts';
import { AppAccessorsInstance } from '../lib/accessors/mod.ts';
import { require } from '../lib/require.ts';
import createRoom from '../lib/roomFactory.ts';
// For some reason Deno couldn't understand the typecast to the original interfaces and said it wasn't a constructor type
const { SlashCommandContext } = require('@rocket.chat/apps-engine/definition/slashcommands/SlashCommandContext.js');
export default async function slashCommandHandler(call, params) {
  const [, commandName, method] = call.split(':');
  const command = AppObjectRegistry.get(`slashcommand:${commandName}`);
  if (!command) {
    return new JsonRpcError(`Slashcommand ${commandName} not found`, -32000);
  }
  let result;
  // If the command is registered, we're pretty safe to assume the app is not undefined
  const app = AppObjectRegistry.get('app');
  app.getLogger().debug(`${commandName}'s ${method} is being executed...`, params);
  try {
    if (method === 'executor' || method === 'previewer') {
      result = await handleExecutor({
        AppAccessorsInstance
      }, command, method, params);
    } else if (method === 'executePreviewItem') {
      result = await handlePreviewItem({
        AppAccessorsInstance
      }, command, params);
    } else {
      return new JsonRpcError(`Method ${method} not found on slashcommand ${commandName}`, -32000);
    }
    app.getLogger().debug(`${commandName}'s ${method} was successfully executed.`);
  } catch (error) {
    app.getLogger().debug(`${commandName}'s ${method} was unsuccessful.`);
    return new JsonRpcError(error.message, -32000);
  }
  return result;
}
/**
 * @param deps Dependencies that need to be injected into the slashcommand
 * @param command The slashcommand that is being executed
 * @param method The method that is being executed
 * @param params The parameters that are being passed to the method
 */ export function handleExecutor(deps, command, method, params) {
  const executor = command[method];
  if (typeof executor !== 'function') {
    throw new Error(`Method ${method} not found on slashcommand ${command.command}`);
  }
  if (!Array.isArray(params) || typeof params[0] !== 'object' || !params[0]) {
    throw new Error(`First parameter must be an object`);
  }
  const { sender, room, params: args, threadId, triggerId } = params[0];
  const context = new SlashCommandContext(sender, createRoom(room, deps.AppAccessorsInstance.getSenderFn()), args, threadId, triggerId);
  return executor.apply(command, [
    context,
    deps.AppAccessorsInstance.getReader(),
    deps.AppAccessorsInstance.getModifier(),
    deps.AppAccessorsInstance.getHttp(),
    deps.AppAccessorsInstance.getPersistence()
  ]);
}
/**
 * @param deps Dependencies that need to be injected into the slashcommand
 * @param command The slashcommand that is being executed
 * @param params The parameters that are being passed to the method
 */ export function handlePreviewItem(deps, command, params) {
  if (typeof command.executePreviewItem !== 'function') {
    throw new Error(`Method  not found on slashcommand ${command.command}`);
  }
  if (!Array.isArray(params) || typeof params[0] !== 'object' || !params[0]) {
    throw new Error(`First parameter must be an object`);
  }
  const [previewItem, { sender, room, params: args, threadId, triggerId }] = params;
  const context = new SlashCommandContext(sender, createRoom(room, deps.AppAccessorsInstance.getSenderFn()), args, threadId, triggerId);
  return command.executePreviewItem(previewItem, context, deps.AppAccessorsInstance.getReader(), deps.AppAccessorsInstance.getModifier(), deps.AppAccessorsInstance.getHttp(), deps.AppAccessorsInstance.getPersistence());
}
//# sourceMappingURL=data:application/json;base64,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