import { MessageBuilder } from '../builders/MessageBuilder.ts';
import { RoomBuilder } from '../builders/RoomBuilder.ts';
import { AppObjectRegistry } from '../../../AppObjectRegistry.ts';
import { require } from '../../../lib/require.ts';
import { formatErrorResponse } from '../formatResponseErrorHandler.ts';
const { UIHelper } = require('@rocket.chat/apps-engine/server/misc/UIHelper.js');
const { RoomType } = require('@rocket.chat/apps-engine/definition/rooms/RoomType.js');
const { RocketChatAssociationModel } = require('@rocket.chat/apps-engine/definition/metadata/RocketChatAssociations.js');
export class ModifyUpdater {
  senderFn;
  constructor(senderFn){
    this.senderFn = senderFn;
  }
  getLivechatUpdater() {
    return new Proxy({
      __kind: 'getLivechatUpdater'
    }, {
      get: (_target, prop)=>(...params)=>prop === 'toJSON' ? {} : this.senderFn({
            method: `accessor:getModifier:getUpdater:getLivechatUpdater:${prop}`,
            params
          }).then((response)=>response.result).catch((err)=>{
            throw formatErrorResponse(err);
          })
    });
  }
  getUserUpdater() {
    return new Proxy({
      __kind: 'getUserUpdater'
    }, {
      get: (_target, prop)=>(...params)=>prop === 'toJSON' ? {} : this.senderFn({
            method: `accessor:getModifier:getUpdater:getUserUpdater:${prop}`,
            params
          }).then((response)=>response.result).catch((err)=>{
            throw formatErrorResponse(err);
          })
    });
  }
  async message(messageId, editor) {
    const response = await this.senderFn({
      method: 'bridges:getMessageBridge:doGetById',
      params: [
        messageId,
        AppObjectRegistry.get('id')
      ]
    }).catch((err)=>{
      throw formatErrorResponse(err);
    });
    const builder = new MessageBuilder(response.result);
    builder.setEditor(editor);
    return builder;
  }
  async room(roomId, _updater) {
    const response = await this.senderFn({
      method: 'bridges:getRoomBridge:doGetById',
      params: [
        roomId,
        AppObjectRegistry.get('id')
      ]
    }).catch((err)=>{
      throw formatErrorResponse(err);
    });
    return new RoomBuilder(response.result);
  }
  finish(builder) {
    switch(builder.kind){
      case RocketChatAssociationModel.MESSAGE:
        return this._finishMessage(builder);
      case RocketChatAssociationModel.ROOM:
        return this._finishRoom(builder);
      default:
        throw new Error('Invalid builder passed to the ModifyUpdater.finish function.');
    }
  }
  async _finishMessage(builder) {
    const result = builder.getMessage();
    if (!result.id) {
      throw new Error("Invalid message, can't update a message without an id.");
    }
    if (!result.sender?.id) {
      throw new Error('Invalid sender assigned to the message.');
    }
    if (result.blocks?.length) {
      result.blocks = UIHelper.assignIds(result.blocks, AppObjectRegistry.get('id') || '');
    }
    const changes = {
      id: result.id,
      ...builder.getChanges()
    };
    await this.senderFn({
      method: 'bridges:getMessageBridge:doUpdate',
      params: [
        changes,
        AppObjectRegistry.get('id')
      ]
    }).catch((err)=>{
      throw formatErrorResponse(err);
    });
  }
  async _finishRoom(builder) {
    const room = builder.getRoom();
    if (!room.id) {
      throw new Error("Invalid room, can't update a room without an id.");
    }
    if (!room.type) {
      throw new Error('Invalid type assigned to the room.');
    }
    if (room.type !== RoomType.LIVE_CHAT) {
      if (!room.creator || !room.creator.id) {
        throw new Error('Invalid creator assigned to the room.');
      }
      if (!room.slugifiedName || !room.slugifiedName.trim()) {
        throw new Error('Invalid slugifiedName assigned to the room.');
      }
    }
    if (!room.displayName || !room.displayName.trim()) {
      throw new Error('Invalid displayName assigned to the room.');
    }
    const changes = {
      id: room.id,
      ...builder.getChanges()
    };
    await this.senderFn({
      method: 'bridges:getRoomBridge:doUpdate',
      params: [
        changes,
        builder.getMembersToBeAddedUsernames(),
        AppObjectRegistry.get('id')
      ]
    }).catch((err)=>{
      throw formatErrorResponse(err);
    });
  }
}
//# sourceMappingURL=data:application/json;base64,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