import { AppStatus } from './AppStatus';
export class App {
  info;
  logger;
  accessors;
  status;
  /**
	 * Create a new App, this is called whenever the server starts up and initiates the Apps.
	 * Note, your implementation of this class should call `super(name, id, version)` so we have it.
	 * Also, please use the `initialize()` method to do items instead of the constructor as the constructor
	 * *might* be called more than once but the `initialize()` will only be called once.
	 */ constructor(info, logger, accessors){
    this.info = info;
    this.logger = logger;
    this.accessors = accessors;
    this.status = AppStatus.UNKNOWN;
    this.logger.debug(`Constructed the App ${this.info.name} (${this.info.id})`, `v${this.info.version} which depends on the API v${this.info.requiredApiVersion}!`, `Created by ${this.info.author.name}`);
    this.setStatus(AppStatus.CONSTRUCTED);
  }
  async getStatus() {
    return this.status;
  }
  /**
	 * Get the name of this App.
	 *
	 * @return {string} the name
	 */ getName() {
    return this.info.name;
  }
  /**
	 * Gets the sluggified name of this App.
	 *
	 * @return {string} the name slugged
	 */ getNameSlug() {
    return this.info.nameSlug;
  }
  /**
	 * Gets the username of this App's app user.
	 *
	 * @return {string} the username of the app user
	 *
	 * @deprecated This method will be removed in the next major version.
	 * Please use read.getUserReader().getAppUser() instead.
	 */ getAppUserUsername() {
    return `${this.info.nameSlug}.bot`;
  }
  /**
	 * Get the ID of this App, please see <link> for how to obtain an ID for your App.
	 *
	 * @return {number} the ID
	 */ getID() {
    return this.info.id;
  }
  /**
	 * Get the version of this App, using http://semver.org/.
	 *
	 * @return {string} the version
	 */ getVersion() {
    return this.info.version;
  }
  /**
	 * Get the description of this App, mostly used to show to the clients/administrators.
	 *
	 * @return {string} the description
	 */ getDescription() {
    return this.info.description;
  }
  /**
	 * Gets the API Version which this App depends on (http://semver.org/).
	 * This property is used for the dependency injections.
	 *
	 * @return {string} the required api version
	 */ getRequiredApiVersion() {
    return this.info.requiredApiVersion;
  }
  /**
	 * Gets the information regarding the author/maintainer of this App.
	 *
	 * @return author information
	 */ getAuthorInfo() {
    return this.info.author;
  }
  /**
	 * Gets the entirity of the App's information.
	 *
	 * @return App information
	 */ getInfo() {
    return this.info;
  }
  /**
	 * Gets the ILogger instance for this App.
	 *
	 * @return the logger instance
	 */ getLogger() {
    return this.logger;
  }
  getAccessors() {
    return this.accessors;
  }
  /**
	 * Method which will be called when the App is initialized. This is the recommended place
	 * to add settings and slash commands. If an error is thrown, all commands will be unregistered.
	 */ async initialize(configurationExtend, environmentRead) {
    await this.extendConfiguration(configurationExtend, environmentRead);
  }
  /**
	 * Method which is called when this App is enabled and can be called several
	 * times during this instance's life time. Once after the `initialize()` is called,
	 * pending it doesn't throw an error, and then anytime the App is enabled by the user.
	 * If this method, `onEnable()`, returns false, then this App will not
	 * actually be enabled (ex: a setting isn't configured).
	 *
	 * @return whether the App should be enabled or not
	 */ async onEnable(environment, configurationModify) {
    return true;
  }
  /**
	 * Method which is called when this App is disabled and it can be called several times.
	 * If this App was enabled and then the user disabled it, this method will be called.
	 */ async onDisable(configurationModify) {}
  /**
	 * Method which is called when the App is uninstalled and it is called one single time.
	 *
	 * This method will NOT be called when an App is getting disabled manually, ONLY when
	 * it's being uninstalled from Rocket.Chat.
	 */ async onUninstall(context, read, http, persistence, modify) {}
  /**
	 * Method which is called when the App is installed and it is called one single time.
	 *
	 * This method is NOT called when the App is updated.
	 */ async onInstall(context, read, http, persistence, modify) {}
  /**
	 * Method which is called when the App is updated and it is called one single time.
	 *
	 * This method is NOT called when the App is installed.
	 */ async onUpdate(context, read, http, persistence, modify) {}
  /**
	 * Method which is called whenever a setting which belongs to this App has been updated
	 * by an external system and not this App itself. The setting passed is the newly updated one.
	 *
	 * @param setting the setting which was updated
	 * @param configurationModify the accessor to modifiy the system
	 * @param reader the reader accessor
	 * @param http an accessor to the outside world
	 */ async onSettingUpdated(setting, configurationModify, read, http) {}
  /**
	 * Method which is called before a setting which belongs to this App is going to be updated
	 * by an external system and not this App itself. The setting passed is the newly updated one.
	 *
	 * @param setting the setting which is going to be updated
	 * @param configurationModify the accessor to modifiy the system
	 * @param reader the reader accessor
	 * @param http an accessor to the outside world
	 */ async onPreSettingUpdate(context, configurationModify, read, http) {
    return context.newSetting;
  }
  /**
	 * Method will be called during initialization. It allows for adding custom configuration options and defaults
	 * @param configuration
	 */ async extendConfiguration(configuration, environmentRead) {}
  /**
	 * Sets the status this App is now at, use only when 100% true (it's protected for a reason).
	 *
	 * @param status the new status of this App
	 */ async setStatus(status) {
    this.logger.debug(`The status is now: ${status}`);
    this.status = status;
  }
  // Avoid leaking references if object is serialized (e.g. to be sent over IPC)
  toJSON() {
    return this.info;
  }
}
//# sourceMappingURL=data:application/json;base64,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