"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LocalBroker = void 0;
const events_1 = require("events");
const logger_1 = require("@rocket.chat/logger");
const models_1 = require("@rocket.chat/models");
const tracing_1 = require("@rocket.chat/tracing");
const _1 = require(".");
const logger = new logger_1.Logger('LocalBroker');
const INTERVAL = 1000;
const TIMEOUT = INTERVAL * 10;
class LocalBroker {
    constructor() {
        this.started = false;
        this.methods = new Map();
        this.events = new events_1.EventEmitter();
        this.services = new Map();
        this.pendingServices = new Set();
        this.defaultDependencies = ['settings'];
    }
    async call(method, data, options) {
        if (options) {
            logger.warn('Options are not supported in LocalBroker');
        }
        return (0, tracing_1.tracerActiveSpan)(`action ${method}`, {}, () => {
            return _1.asyncLocalStorage.run({
                id: 'ctx.id',
                nodeID: 'ctx.nodeID',
                requestID: 'ctx.requestID',
                broker: this,
            }, () => this.methods.get(method)?.(...data));
        }, (0, tracing_1.injectCurrentContext)());
    }
    async destroyService(instance) {
        const namespace = instance.getName();
        instance.getEvents().forEach((event) => event.listeners.forEach((listener) => this.events.removeListener(event.eventName, listener)));
        const methods = instance.constructor?.name === 'Object'
            ? Object.getOwnPropertyNames(instance)
            : Object.getOwnPropertyNames(Object.getPrototypeOf(instance));
        for (const method of methods) {
            if (method === 'constructor') {
                continue;
            }
            this.methods.delete(`${namespace}.${method}`);
        }
        instance.removeAllListeners();
        await instance.stopped();
        this.services.delete(namespace);
    }
    /**
     * Creates a service and adds it to the local broker. In case of the broker is already started, it will start the service automatically.
     */
    createService(instance, serviceDependencies = []) {
        const serviceName = instance.getName();
        if (!serviceName || serviceName === '') {
            throw new Error('Service name cannot be empty');
        }
        if (this.services.has(serviceName)) {
            throw new Error(`Service ${serviceName} already exists`);
        }
        // TODO: find a better way to handle default dependencies and avoid loops
        const dependencies = [...serviceDependencies, ...(serviceName === 'settings' ? [] : this.defaultDependencies)].filter((dependency) => dependency !== serviceName);
        instance.created();
        instance.getEvents().forEach((event) => event.listeners.forEach((listener) => this.events.on(event.eventName, listener)));
        const methods = instance.constructor?.name === 'Object'
            ? Object.getOwnPropertyNames(instance)
            : Object.getOwnPropertyNames(Object.getPrototypeOf(instance));
        for (const method of methods) {
            if (method === 'constructor') {
                continue;
            }
            const i = instance;
            this.methods.set(`${serviceName}.${method}`, i[method].bind(i));
        }
        this.services.set(serviceName, { instance, dependencies, isStarted: false });
        this.registerPendingServices(Array.from(new Set([serviceName, ...dependencies])));
        if (this.started) {
            void this.start();
        }
    }
    onBroadcast(callback) {
        this.events.on('broadcast', callback);
    }
    async broadcast(event, ...args) {
        this.broadcastLocal(event, ...args);
        this.events.emit('broadcast', event, args);
    }
    async broadcastLocal(event, ...args) {
        this.events.emit(event, ...args);
    }
    async broadcastToServices(_services, event, ...args) {
        this.events.emit(event, ...args);
    }
    async nodeList() {
        // TODO models should not be called form here. we should create an abstraction to an internal service to perform this query
        const instances = await models_1.InstanceStatus.find({}, { projection: { _id: 1 } }).toArray();
        return instances.map(({ _id }) => ({ id: _id, available: true }));
    }
    /**
     * Registers services to be started. We're assuming that each service will only have one level of dependencies.
     */
    registerPendingServices(services = []) {
        services
            .filter((e) => !this.services.has(e) || !this.services.get(e)?.isStarted)
            .forEach((service) => this.pendingServices.add(service));
    }
    /**
     * Removes a service from the pending services set.
     */
    removePendingService(service) {
        this.pendingServices.delete(service);
    }
    async startService(service) {
        const serviceName = service.instance.getName();
        if (typeof service === 'string') {
            logger.debug(`Service ${serviceName} is not in the services map. Bringing it back to queue`);
            return;
        }
        if (service?.isStarted) {
            logger.debug(`Service ${serviceName} already started`);
            return;
        }
        const pendingDependencies = service.dependencies.filter((e) => !this.services.has(e) || !this.services.get(e)?.isStarted);
        if (pendingDependencies.length > 0) {
            logger.debug(`Service ${serviceName} has dependencies that are not started yet, bringing it back to queue: ${pendingDependencies.join(', ')}`);
            return;
        }
        await service.instance.started();
        this.services.set(serviceName, { ...service, isStarted: true });
        this.removePendingService(serviceName);
        logger.debug(`Service ${serviceName} successfully started`);
    }
    async start() {
        const startTime = Date.now();
        return new Promise((resolve, reject) => {
            const intervalId = setInterval(async () => {
                const elapsed = Date.now() - startTime;
                if (this.pendingServices.size === 0) {
                    const availableServices = Array.from(this.services.values()).filter((service) => service.isStarted);
                    logger.info(`All ${availableServices.length} services available`);
                    clearInterval(intervalId);
                    return resolve();
                }
                if (elapsed > TIMEOUT) {
                    clearInterval(intervalId);
                    const pendingServices = Array.from(this.pendingServices).join(', ');
                    const error = new Error(`Timeout while waiting for LocalBroker services: ${pendingServices}`);
                    logger.error(error);
                    return reject(error);
                }
                for await (const service of Array.from(this.pendingServices)) {
                    const serviceInstance = this.services.get(service);
                    if (serviceInstance) {
                        await this.startService(serviceInstance);
                    }
                }
                logger.debug(`Waiting for ${this.pendingServices.size} pending services`);
            }, INTERVAL);
        });
    }
}
exports.LocalBroker = LocalBroker;
//# sourceMappingURL=LocalBroker.js.map