"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cronJobs = exports.AgendaCronJobs = void 0;
const agenda_1 = require("@rocket.chat/agenda");
const models_1 = require("@rocket.chat/models");
const random_1 = require("@rocket.chat/random");
const runCronJobFunctionAndPersistResult = async (fn, jobName) => {
    const { insertedId } = await models_1.CronHistory.insertOne({
        _id: random_1.Random.id(),
        intendedAt: new Date(),
        name: jobName,
        startedAt: new Date(),
    });
    try {
        const result = await fn();
        await models_1.CronHistory.updateOne({ _id: insertedId }, {
            $set: {
                finishedAt: new Date(),
                result,
            },
        });
        return result;
    }
    catch (error) {
        await models_1.CronHistory.updateOne({ _id: insertedId }, {
            $set: {
                finishedAt: new Date(),
                error: error?.stack ? error.stack : error,
            },
        });
        throw error;
    }
};
class AgendaCronJobs {
    constructor() {
        this.reservedJobs = [];
    }
    get started() {
        return Boolean(this.scheduler);
    }
    async start(mongo) {
        this.scheduler = new agenda_1.Agenda({
            mongo,
            db: { collection: 'rocketchat_cron' },
            defaultConcurrency: 1,
            processEvery: '1 minute',
        });
        await this.scheduler.start();
        for await (const job of this.reservedJobs) {
            if (job.timestamped) {
                await this.addAtTimestamp(job.name, job.when, job.callback);
            }
            else {
                await this.add(job.name, job.schedule, job.callback);
            }
        }
        this.reservedJobs = [];
    }
    async add(name, schedule, callback) {
        if (!this.scheduler) {
            return this.reserve({ name, schedule, callback, timestamped: false });
        }
        await this.define(name, callback);
        await this.scheduler.every(schedule, name, {}, {});
    }
    async addAtTimestamp(name, when, callback) {
        if (!this.scheduler) {
            return this.reserve({ name, when, callback, timestamped: true });
        }
        await this.define(name, callback);
        await this.scheduler.schedule(when, name, {});
    }
    async remove(name) {
        if (!this.scheduler) {
            return this.unreserve(name);
        }
        await this.scheduler.cancel({ name });
    }
    async has(jobName) {
        if (!this.scheduler) {
            return Boolean(this.reservedJobs.find(({ name }) => name === jobName));
        }
        return this.scheduler.has({ name: jobName });
    }
    async reserve(config) {
        this.reservedJobs = [...this.reservedJobs, config];
    }
    async unreserve(jobName) {
        this.reservedJobs = this.reservedJobs.filter(({ name }) => name !== jobName);
    }
    async define(jobName, callback) {
        if (!this.scheduler) {
            throw new Error('Scheduler is not running.');
        }
        this.scheduler.define(jobName, async () => {
            await runCronJobFunctionAndPersistResult(async () => callback(), jobName);
        });
    }
}
exports.AgendaCronJobs = AgendaCronJobs;
exports.cronJobs = new AgendaCronJobs();
//# sourceMappingURL=index.js.map