"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.member = member;
const core_services_1 = require("@rocket.chat/core-services");
const federation_sdk_1 = require("@rocket.chat/federation-sdk");
const logger_1 = require("@rocket.chat/logger");
const models_1 = require("@rocket.chat/models");
const FederationMatrix_1 = require("../FederationMatrix");
const logger = new logger_1.Logger('federation-matrix:member');
async function membershipLeaveAction(event) {
    const room = await models_1.Rooms.findOne({ 'federation.mrid': event.room_id }, { projection: { _id: 1 } });
    if (!room) {
        logger.warn(`No bridged room found for Matrix room_id: ${event.room_id}`);
        return;
    }
    const serverName = federation_sdk_1.federationSDK.getConfig('serverName');
    const [affectedUsername] = (0, FederationMatrix_1.getUsernameServername)(event.state_key, serverName);
    // state_key is the user affected by the membership change
    const affectedUser = await models_1.Users.findOneByUsername(affectedUsername);
    if (!affectedUser) {
        logger.error(`No Rocket.Chat user found for bridged user: ${event.state_key}`);
        return;
    }
    // Check if this is a kick (sender != state_key) or voluntary leave (sender == state_key)
    if (event.sender === event.state_key) {
        // Voluntary leave
        await core_services_1.Room.removeUserFromRoom(room._id, affectedUser);
        logger.info(`User ${affectedUser.username} left room ${room._id} via Matrix federation`);
    }
    else {
        // Kick - find who kicked
        const [kickerUsername] = (0, FederationMatrix_1.getUsernameServername)(event.sender, serverName);
        const kickerUser = await models_1.Users.findOneByUsername(kickerUsername);
        await core_services_1.Room.removeUserFromRoom(room._id, affectedUser, {
            byUser: kickerUser || { _id: 'matrix.federation', username: 'Matrix User' },
        });
        const reasonText = event.content.reason ? ` Reason: ${event.content.reason}` : '';
        logger.info(`User ${affectedUser.username} was kicked from room ${room._id} by ${event.sender} via Matrix federation.${reasonText}`);
    }
}
async function membershipJoinAction(event) {
    const room = await models_1.Rooms.findOne({ 'federation.mrid': event.room_id });
    if (!room) {
        logger.warn(`No bridged room found for room_id: ${event.room_id}`);
        return;
    }
    const [username, serverName, isLocal] = (0, FederationMatrix_1.getUsernameServername)(event.sender, federation_sdk_1.federationSDK.getConfig('serverName'));
    // for local users we must to remove the @ and the server domain
    const localUser = isLocal && (await models_1.Users.findOneByUsername(username));
    if (localUser) {
        const subscription = await models_1.Subscriptions.findOneByRoomIdAndUserId(room._id, localUser._id);
        if (subscription) {
            return;
        }
        await core_services_1.Room.addUserToRoom(room._id, localUser);
        return;
    }
    if (!serverName) {
        throw new Error('Invalid sender format, missing server name');
    }
    const insertedId = await (0, FederationMatrix_1.createOrUpdateFederatedUser)({
        username: event.state_key,
        origin: serverName,
        name: event.content.displayname || event.state_key,
    });
    const user = await models_1.Users.findOneById(insertedId);
    if (!user) {
        console.warn(`User with ID ${insertedId} not found after insertion`);
        return;
    }
    await core_services_1.Room.addUserToRoom(room._id, user);
}
function member(emitter) {
    emitter.on('homeserver.matrix.membership', async ({ event }) => {
        try {
            if (event.content.membership === 'leave') {
                return membershipLeaveAction(event);
            }
            if (event.content.membership === 'join') {
                return membershipJoinAction(event);
            }
            logger.debug(`Ignoring membership event with membership: ${event.content.membership}`);
        }
        catch (error) {
            logger.error('Failed to process Matrix membership event:', error);
        }
    });
}
//# sourceMappingURL=member.js.map