"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.configureFederationMatrixSettings = configureFederationMatrixSettings;
exports.setupFederationMatrix = setupFederationMatrix;
const emitter_1 = require("@rocket.chat/emitter");
const federation_sdk_1 = require("@rocket.chat/federation-sdk");
const logger_1 = require("@rocket.chat/logger");
const events_1 = require("./events");
const logger = new logger_1.Logger('FederationSetup');
function validateDomain(domain) {
    const value = domain.trim();
    if (!value) {
        logger.error('The Federation domain is not set');
        return false;
    }
    if (value.toLowerCase() !== value) {
        logger.error(`The Federation domain "${value}" cannot have uppercase letters`);
        return false;
    }
    try {
        const valid = new URL(`https://${value}`).hostname === value;
        if (!valid) {
            throw new Error();
        }
    }
    catch {
        logger.error(`The configured Federation domain "${value}" is not valid`);
        return false;
    }
    return true;
}
function configureFederationMatrixSettings(settings) {
    const { instanceId, domain: serverName, signingKey, signingAlgorithm: signingAlg, signingVersion, allowedEncryptedRooms, allowedNonPrivateRooms, processEDUTyping, processEDUPresence, } = settings;
    if (!validateDomain(serverName)) {
        throw new Error('Invalid Federation domain');
    }
    federation_sdk_1.federationSDK.setConfig({
        instanceId,
        serverName,
        keyRefreshInterval: Number.parseInt(process.env.MATRIX_KEY_REFRESH_INTERVAL || '60', 10),
        matrixDomain: serverName,
        version: process.env.SERVER_VERSION || '1.0',
        port: Number.parseInt(process.env.SERVER_PORT || '8080', 10),
        signingKey: `${signingAlg} ${signingVersion} ${signingKey}`,
        signingKeyPath: '', // TODO remove
        media: {
            maxFileSize: Number.parseInt(process.env.MEDIA_MAX_FILE_SIZE || '100', 10) * 1024 * 1024,
            allowedMimeTypes: process.env.MEDIA_ALLOWED_MIME_TYPES?.split(',') || [
                'image/jpeg',
                'image/png',
                'image/gif',
                'image/webp',
                'text/plain',
                'application/pdf',
                'video/mp4',
                'audio/mpeg',
                'audio/ogg',
            ],
            enableThumbnails: process.env.MEDIA_ENABLE_THUMBNAILS !== 'true',
            rateLimits: {
                uploadPerMinute: Number.parseInt(process.env.MEDIA_UPLOAD_RATE_LIMIT || '10', 10),
                downloadPerMinute: Number.parseInt(process.env.MEDIA_DOWNLOAD_RATE_LIMIT || '60', 10),
            },
        },
        invite: {
            allowedEncryptedRooms,
            allowedNonPrivateRooms,
        },
        edu: {
            processTyping: processEDUTyping,
            processPresence: processEDUPresence,
        },
    });
}
async function setupFederationMatrix() {
    const eventHandler = new emitter_1.Emitter();
    await (0, federation_sdk_1.init)({
        emitter: eventHandler,
        dbConfig: {
            uri: process.env.MONGO_URL || 'mongodb://localhost:3001/meteor',
            poolSize: Number.parseInt(process.env.DATABASE_POOL_SIZE || '10', 10),
        },
    });
    (0, events_1.registerEvents)(eventHandler);
}
//# sourceMappingURL=setup.js.map