"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatsTokenBuilder = exports.MockedLicenseBuilder = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const token_1 = require("./token");
class MockedLicenseBuilder {
    constructor() {
        this.grantedModules = [];
        this.limits = {};
        this.information = {
            autoRenew: true,
            // expires in 1 year
            visualExpiration: new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toISOString(),
            // 15 days before expiration
            notifyAdminsAt: new Date(new Date().setDate(new Date().getDate() + 15)).toISOString(),
            // 30 days before expiration
            notifyUsersAt: new Date(new Date().setDate(new Date().getDate() + 30)).toISOString(),
            trial: false,
            offline: false,
            createdAt: new Date().toISOString(),
            grantedBy: {
                method: 'manual',
                seller: 'Rocket.Cat',
            },
            tags: [
                {
                    name: 'Test',
                    color: 'blue',
                },
            ],
        };
        this.validation = {
            serverUrls: [
                {
                    value: 'localhost:3000',
                    type: 'url',
                },
            ],
            serverVersions: [
                {
                    value: '3.0.0',
                },
            ],
            serverUniqueId: '1234567890',
            cloudWorkspaceId: '1234567890',
            validPeriods: [
                {
                    invalidBehavior: 'disable_modules',
                    modules: ['livechat-enterprise'],
                    validFrom: new Date(new Date().setFullYear(new Date().getFullYear() - 1)).toISOString(),
                    validUntil: new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toISOString(),
                },
            ],
            statisticsReport: {
                required: true,
                allowedStaleInDays: 30,
            },
        };
    }
    withExpiredDate() {
        // expired 1 minute ago
        const date = new Date();
        date.setMinutes(date.getMinutes() - 1);
        const expired = date.toISOString();
        const rule = this.validation.validPeriods.find((period) => period.invalidBehavior === 'invalidate_license');
        if (rule) {
            rule.validUntil = expired;
        }
        else {
            this.validation.validPeriods.push({
                invalidBehavior: 'invalidate_license',
                validFrom: new Date(new Date().setFullYear(new Date().getFullYear() - 1)).toISOString(),
                validUntil: expired,
            });
        }
        return this;
    }
    withNotStartedDate() {
        // starts in 1 minute
        const date = new Date();
        date.setMinutes(date.getMinutes() + 1);
        const starts = date.toISOString();
        const rule = this.validation.validPeriods.find((period) => period.invalidBehavior === 'invalidate_license');
        if (rule) {
            rule.validFrom = starts;
        }
        else {
            this.validation.validPeriods.push({
                invalidBehavior: 'invalidate_license',
                validUntil: new Date(new Date().setFullYear(new Date().getFullYear() + 1)).toISOString(),
                validFrom: starts,
            });
        }
        return this;
    }
    resetValidPeriods() {
        this.validation.validPeriods = [];
        return this;
    }
    withValidPeriod(period) {
        this.validation.validPeriods.push(period);
        return this;
    }
    withGrantedTo(grantedTo) {
        this.information.grantedTo = grantedTo;
        return this;
    }
    withServerUrls(urls) {
        this.validation.serverUrls = this.validation.serverUrls ?? [];
        this.validation.serverUrls.push(urls);
        return this;
    }
    withServerVersions(versions) {
        this.validation.serverVersions = this.validation.serverVersions ?? [];
        this.validation.serverVersions.push(versions);
        return this;
    }
    withGrantedModules(modules) {
        this.grantedModules = this.grantedModules ?? [];
        this.grantedModules.push(...modules.map((module) => ({ module, external: !core_typings_1.CoreModules.includes(module) })));
        return this;
    }
    withNoGrantedModules(modules) {
        this.grantedModules = this.grantedModules ?? [];
        this.grantedModules = this.grantedModules.filter(({ module }) => !modules.includes(module));
        return this;
    }
    withLimits(key, value) {
        this.limits = this.limits ?? {};
        this.limits[key] = value;
        return this;
    }
    build() {
        return {
            version: '3.0',
            information: this.information,
            validation: this.validation,
            grantedModules: [...new Set(this.grantedModules)],
            limits: {
                activeUsers: [],
                guestUsers: [],
                roomsPerGuest: [],
                privateApps: [],
                marketplaceApps: [],
                monthlyActiveContacts: [],
                ...this.limits,
            },
            cloudMeta: this.cloudMeta,
        };
    }
    sign() {
        return (0, token_1.encrypt)(this.build());
    }
}
exports.MockedLicenseBuilder = MockedLicenseBuilder;
class StatsTokenBuilder {
    constructor() {
        this.token = {
            workspaceId: '123456789',
            uniqueId: '123456789',
            recordId: '123456789',
            timestamp: new Date().toISOString(),
            info: {},
        };
    }
    withTimeStamp(date) {
        this.token.timestamp = date.toISOString();
        return this;
    }
    build() {
        return this.token;
    }
    sign() {
        return (0, token_1.encryptStatsToken)(this.token);
    }
}
exports.StatsTokenBuilder = StatsTokenBuilder;
//# sourceMappingURL=MockedLicenseBuilder.js.map