"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseManager = void 0;
const emitter_1 = require("@rocket.chat/emitter");
const deprecated_1 = require("./deprecated");
const DuplicatedLicenseError_1 = require("./errors/DuplicatedLicenseError");
const InvalidLicenseError_1 = require("./errors/InvalidLicenseError");
const NotReadyForValidation_1 = require("./errors/NotReadyForValidation");
const emitter_2 = require("./events/emitter");
const logger_1 = require("./logger");
const modules_1 = require("./modules");
const pendingLicense_1 = require("./pendingLicense");
const tags_1 = require("./tags");
const token_1 = require("./token");
const convertToV3_1 = require("./v2/convertToV3");
const filterBehaviorsResult_1 = require("./validation/filterBehaviorsResult");
const getCurrentValueForLicenseLimit_1 = require("./validation/getCurrentValueForLicenseLimit");
const getModulesToDisable_1 = require("./validation/getModulesToDisable");
const isBehaviorsInResult_1 = require("./validation/isBehaviorsInResult");
const isReadyForValidation_1 = require("./validation/isReadyForValidation");
const runValidation_1 = require("./validation/runValidation");
const validateDefaultLimits_1 = require("./validation/validateDefaultLimits");
const validateFormat_1 = require("./validation/validateFormat");
const validateLicenseLimits_1 = require("./validation/validateLicenseLimits");
const globalLimitKinds = ['activeUsers', 'guestUsers', 'privateApps', 'marketplaceApps', 'monthlyActiveContacts'];
class LicenseManager extends emitter_1.Emitter {
    constructor() {
        super(...arguments);
        this.dataCounters = new Map();
        this.pendingLicense = '';
        this.tags = new Set();
        this.modules = new Set();
        this.states = new Map();
    }
    get shouldPreventActionResults() {
        const state = this.states.get('prevent_action') ?? new Map();
        this.states.set('prevent_action', state);
        return state;
    }
    get license() {
        return this._license;
    }
    get unmodifiedLicense() {
        return this._unmodifiedLicense;
    }
    get valid() {
        return this._valid;
    }
    get encryptedLicense() {
        if (!this.hasValidLicense()) {
            return undefined;
        }
        return this._lockedLicense;
    }
    async setWorkspaceUrl(url) {
        this.workspaceUrl = url.replace(/\/$/, '').replace(/^https?:\/\/(.*)$/, '$1');
        if (pendingLicense_1.hasPendingLicense.call(this)) {
            await pendingLicense_1.applyPendingLicense.call(this);
        }
    }
    getWorkspaceUrl() {
        return this.workspaceUrl;
    }
    async revalidateLicense(options = {}) {
        if (!this.hasValidLicense()) {
            return;
        }
        try {
            await this.validateLicense({ ...options, isNewLicense: false, triggerSync: true });
        }
        catch (e) {
            if (e instanceof InvalidLicenseError_1.InvalidLicenseError) {
                this.invalidateLicense();
                this.emit('sync');
            }
        }
    }
    /**
     * The sync method should be called when a license from a different instance is has changed, so the local instance
     * needs to be updated. This method will validate the license and update the local instance if the license is valid, but will not trigger the onSync event.
     */
    async sync(options = {}) {
        if (!this.hasValidLicense()) {
            return;
        }
        try {
            await this.validateLicense({ ...options, isNewLicense: false, triggerSync: false });
        }
        catch (e) {
            if (e instanceof InvalidLicenseError_1.InvalidLicenseError) {
                this.invalidateLicense();
            }
        }
    }
    clearLicenseData() {
        this._license = undefined;
        this._unmodifiedLicense = undefined;
        this._valid = false;
        this._lockedLicense = undefined;
        this.states.clear();
        pendingLicense_1.clearPendingLicense.call(this);
    }
    invalidateLicense() {
        this._valid = false;
        this.states.clear();
        modules_1.invalidateAll.call(this);
        emitter_2.licenseInvalidated.call(this);
    }
    remove() {
        if (!this._license) {
            return;
        }
        this.clearLicenseData();
        modules_1.invalidateAll.call(this);
        this.emit('removed');
    }
    async setLicenseV3(newLicense, encryptedLicense, originalLicense, isNewLicense) {
        const hadValidLicense = this.hasValidLicense();
        this.clearLicenseData();
        try {
            this._unmodifiedLicense = originalLicense || newLicense;
            this._license = newLicense;
            this._lockedLicense = encryptedLicense;
            await this.validateLicense({ isNewLicense });
        }
        catch (e) {
            if (e instanceof InvalidLicenseError_1.InvalidLicenseError) {
                if (hadValidLicense) {
                    this.invalidateLicense();
                }
            }
        }
    }
    async setLicenseV2(newLicense, encryptedLicense, isNewLicense) {
        return this.setLicenseV3((0, convertToV3_1.convertToV3)(newLicense), encryptedLicense, newLicense, isNewLicense);
    }
    isLicenseDuplicated(encryptedLicense) {
        return Boolean(this._lockedLicense && this._lockedLicense === encryptedLicense);
    }
    async validateLicense(options = {
        triggerSync: true,
    }) {
        if (!this._license) {
            throw new InvalidLicenseError_1.InvalidLicenseError();
        }
        if (!isReadyForValidation_1.isReadyForValidation.call(this)) {
            throw new NotReadyForValidation_1.NotReadyForValidation();
        }
        const validationResult = await runValidation_1.runValidation.call(this, this._license, {
            behaviors: ['invalidate_license', 'start_fair_policy', 'prevent_installation', 'disable_modules'],
            ...options,
        });
        if ((0, isBehaviorsInResult_1.isBehaviorsInResult)(validationResult, ['invalidate_license', 'prevent_installation'])) {
            throw new InvalidLicenseError_1.InvalidLicenseError();
        }
        const shouldLogModules = !this._valid || options.isNewLicense;
        this._valid = true;
        if (this._license.information.tags) {
            tags_1.replaceTags.call(this, this._license.information.tags);
        }
        const disabledModules = (0, getModulesToDisable_1.getModulesToDisable)(validationResult);
        const modulesToEnable = this._license.grantedModules.filter(({ module }) => !disabledModules.includes(module));
        const modulesChanged = modules_1.replaceModules.call(this, modulesToEnable.map(({ module }) => module));
        if (shouldLogModules || modulesChanged) {
            logger_1.logger.log({ msg: 'License validated', modules: modulesToEnable });
        }
        if (!options.isNewLicense) {
            this.triggerBehaviorEvents(validationResult);
        }
        emitter_2.licenseValidated.call(this);
        // If something changed in the license and the sync option is enabled, trigger a sync
        if (((!options.isNewLicense &&
            (0, filterBehaviorsResult_1.filterBehaviorsResult)(validationResult, ['invalidate_license', 'start_fair_policy', 'prevent_installation'])) ||
            modulesChanged) &&
            options.triggerSync) {
            this.emit('sync');
        }
    }
    async setLicense(encryptedLicense, isNewLicense = true) {
        if (!(await (0, validateFormat_1.validateFormat)(encryptedLicense))) {
            throw new InvalidLicenseError_1.InvalidLicenseError();
        }
        if (this.isLicenseDuplicated(encryptedLicense)) {
            // If there is a pending license but the user is trying to revert to the license that is currently active
            if (pendingLicense_1.hasPendingLicense.call(this) && !pendingLicense_1.isPendingLicense.call(this, encryptedLicense)) {
                // simply remove the pending license
                pendingLicense_1.clearPendingLicense.call(this);
                throw new Error('Invalid license');
            }
            /**
             * The license can be set with future minimum date, failing during the first set,
             * but if the user tries to set the same license again later it can be valid or not, so we need to check it again
             */
            if (this.hasValidLicense()) {
                throw new DuplicatedLicenseError_1.DuplicatedLicenseError();
            }
        }
        if (!isReadyForValidation_1.isReadyForValidation.call(this)) {
            // If we can't validate the license data yet, but is a valid license string, store it to validate when we can
            pendingLicense_1.setPendingLicense.call(this, encryptedLicense);
            throw new NotReadyForValidation_1.NotReadyForValidation();
        }
        logger_1.logger.info('New Enterprise License');
        try {
            const decrypted = JSON.parse(await (0, token_1.decrypt)(encryptedLicense));
            logger_1.logger.debug({ msg: 'license', decrypted });
            if (!encryptedLicense.startsWith('RCV3_')) {
                await this.setLicenseV2(decrypted, encryptedLicense, isNewLicense);
                return true;
            }
            await this.setLicenseV3(decrypted, encryptedLicense, decrypted, isNewLicense);
            this.emit('installed');
            return true;
        }
        catch (e) {
            logger_1.logger.error('Invalid license');
            logger_1.logger.error({ msg: 'Invalid raw license', encryptedLicense, e });
            throw new InvalidLicenseError_1.InvalidLicenseError();
        }
    }
    triggerBehaviorEvents(validationResult) {
        for (const { ...options } of validationResult) {
            emitter_2.behaviorTriggered.call(this, { ...options });
        }
    }
    triggerBehaviorEventsToggled(validationResult) {
        for (const { ...options } of validationResult) {
            emitter_2.behaviorTriggeredToggled.call(this, { ...options });
        }
    }
    hasValidLicense() {
        return Boolean(this.getLicense());
    }
    getLicense() {
        if (this._valid && this._license) {
            return this._license;
        }
    }
    syncShouldPreventActionResults(actions) {
        for (const [action, shouldPreventAction] of Object.entries(actions)) {
            this.shouldPreventActionResults.set(action, shouldPreventAction);
        }
    }
    async shouldPreventActionResultsMap() {
        const keys = [
            'activeUsers',
            'guestUsers',
            'roomsPerGuest',
            'privateApps',
            'marketplaceApps',
            'monthlyActiveContacts',
        ];
        const license = this.getLicense();
        const items = await Promise.all(keys.map(async (limit) => {
            const cached = this.shouldPreventActionResults.get(limit);
            if (cached !== undefined) {
                return [limit, cached];
            }
            const fresh = license
                ? (0, isBehaviorsInResult_1.isBehaviorsInResult)(await validateLicenseLimits_1.validateLicenseLimits.call(this, license, {
                    behaviors: ['prevent_action'],
                    limits: [limit],
                }), ['prevent_action'])
                : (0, isBehaviorsInResult_1.isBehaviorsInResult)(await validateDefaultLimits_1.validateDefaultLimits.call(this, { behaviors: ['prevent_action'], limits: [limit] }), [
                    'prevent_action',
                ]);
            this.shouldPreventActionResults.set(limit, fresh);
            return [limit, fresh];
        }));
        return Object.fromEntries(items);
    }
    async shouldPreventAction(action, extraCount = 0, context = {}, { suppressLog } = {
        suppressLog: process.env.LICENSE_VALIDATION_SUPPRESS_LOG !== 'false',
    }) {
        const options = {
            ...(extraCount && { behaviors: ['prevent_action'] }),
            isNewLicense: false,
            suppressLog: !!suppressLog,
            limits: [action],
            context: {
                [action]: {
                    extraCount,
                    ...context,
                },
            },
        };
        const license = this.getLicense();
        if (!license) {
            return (0, isBehaviorsInResult_1.isBehaviorsInResult)(await validateDefaultLimits_1.validateDefaultLimits.call(this, options), ['prevent_action']);
        }
        const validationResult = await runValidation_1.runValidation.call(this, license, options);
        const shouldPreventAction = (0, isBehaviorsInResult_1.isBehaviorsInResult)(validationResult, ['prevent_action']);
        // extra values should not call events since they are not actually reaching the limit just checking if they would
        if (extraCount) {
            return shouldPreventAction;
        }
        // check if any of the behaviors that should trigger a sync changed
        if (['invalidate_license', 'disable_modules', 'start_fair_policy'].some((behavior) => {
            const hasChanged = this.consolidateBehaviorState(action, behavior, (0, isBehaviorsInResult_1.isBehaviorsInResult)(validationResult, [behavior]));
            if (hasChanged && behavior === 'start_fair_policy') {
                this.triggerBehaviorEventsToggled([
                    {
                        behavior: 'start_fair_policy',
                        reason: 'limit',
                        limit: action,
                    },
                ]);
            }
            return hasChanged;
        })) {
            await this.revalidateLicense();
        }
        const eventsToEmit = shouldPreventAction
            ? (0, filterBehaviorsResult_1.filterBehaviorsResult)(validationResult, ['prevent_action'])
            : [
                {
                    behavior: 'allow_action',
                    modules: [],
                    reason: 'limit',
                    limit: action,
                },
            ];
        if (this.consolidateBehaviorState(action, 'prevent_action', shouldPreventAction)) {
            this.triggerBehaviorEventsToggled(eventsToEmit);
        }
        this.triggerBehaviorEvents(eventsToEmit);
        return shouldPreventAction;
    }
    consolidateBehaviorState(action, behavior, triggered) {
        // check if the behavior changed
        const state = this.states.get(behavior) ?? new Map();
        const currentState = state.get(action) ?? false;
        if (currentState === triggered) {
            return false;
        }
        // if it changed, update the state
        state.set(action, triggered);
        this.states.set(behavior, state);
        return true;
    }
    async getInfo({ limits: includeLimits, currentValues: loadCurrentValues, license: includeLicense, }) {
        const activeModules = modules_1.getModules.call(this);
        const externalModules = modules_1.getExternalModules.call(this);
        const license = this.getLicense();
        // Get all limits present in the license and their current value
        const limits = Object.fromEntries((includeLimits &&
            (await Promise.all(globalLimitKinds
                .map((limitKey) => [limitKey, (0, deprecated_1.getLicenseLimit)(license, limitKey)])
                .map(async ([limitKey, max]) => {
                return [
                    limitKey,
                    {
                        ...(loadCurrentValues && { value: await getCurrentValueForLicenseLimit_1.getCurrentValueForLicenseLimit.call(this, limitKey) }),
                        max,
                    },
                ];
            })))) ||
            []);
        return {
            license: (includeLicense && license) || undefined,
            activeModules,
            externalModules,
            preventedActions: await this.shouldPreventActionResultsMap(),
            limits: limits,
            tags: license?.information.tags || [],
            trial: Boolean(license?.information.trial),
        };
    }
}
exports.LicenseManager = LicenseManager;
//# sourceMappingURL=license.js.map