"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateLimits = validateLimits;
const isItemAllowed_1 = require("../isItemAllowed");
const logger_1 = require("../logger");
const getCurrentValueForLicenseLimit_1 = require("./getCurrentValueForLicenseLimit");
const getResultingBehavior_1 = require("./getResultingBehavior");
const logKind_1 = require("./logKind");
const validateLimit_1 = require("./validateLimit");
async function validateLimits(limits, options) {
    const limitKeys = Object.keys(limits).filter((limit) => (0, isItemAllowed_1.isLimitAllowed)(limit, options));
    return (await Promise.all(limitKeys.map(async (limitKey) => {
        // Filter the limit list before running any query in the database so we don't end up loading some value we won't use.
        const limitList = limits[limitKey]?.filter(({ behavior, max }) => max >= 0 && (0, isItemAllowed_1.isBehaviorAllowed)(behavior, options));
        if (!limitList?.length) {
            return [];
        }
        const extraCount = options.context?.[limitKey]?.extraCount ?? 0;
        const currentValue = (await getCurrentValueForLicenseLimit_1.getCurrentValueForLicenseLimit.call(this, limitKey, options.context?.[limitKey])) + extraCount;
        return limitList
            .filter(({ max, behavior }) => (0, validateLimit_1.validateLimit)(max, currentValue, behavior, extraCount))
            .map((limit) => {
            if (!options.suppressLog) {
                switch ((0, logKind_1.logKind)(limit.behavior)) {
                    case 'error':
                        logger_1.logger.error({
                            msg: 'Limit validation failed',
                            kind: limitKey,
                            limit,
                        });
                        break;
                    case 'info':
                        logger_1.logger.info({
                            msg: 'Limit validation failed',
                            kind: limitKey,
                            limit,
                        });
                        break;
                }
            }
            return (0, getResultingBehavior_1.getResultingBehavior)(limit, { reason: 'limit', limit: limitKey });
        });
    }))).flat();
}
//# sourceMappingURL=validateLimits.js.map