"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlobalSignalProcessor = void 0;
const emitter_1 = require("@rocket.chat/emitter");
const media_signaling_1 = require("@rocket.chat/media-signaling");
const models_1 = require("@rocket.chat/models");
const logger_1 = require("../logger");
const CallDirector_1 = require("../server/CallDirector");
const UserActorAgent_1 = require("./agents/UserActorAgent");
const buildNewCallSignal_1 = require("../server/buildNewCallSignal");
const stripSensitiveData_1 = require("../server/stripSensitiveData");
class GlobalSignalProcessor {
    constructor() {
        this.emitter = new emitter_1.Emitter();
    }
    async processSignal(uid, signal) {
        logger_1.logger.debug({ msg: 'GlobalSignalProcessor.processSignal', signal: (0, stripSensitiveData_1.stripSensitiveDataFromSignal)(signal), uid });
        switch (signal.type) {
            case 'register':
                return this.processRegisterSignal(uid, signal);
            case 'request-call':
                return this.processRequestCallSignal(uid, signal);
        }
        if ('callId' in signal) {
            return this.processCallSignal(uid, signal);
        }
        logger_1.logger.error({ msg: 'Unrecognized media signal', signal: (0, stripSensitiveData_1.stripSensitiveDataFromSignal)(signal) });
    }
    sendSignal(toUid, signal) {
        this.emitter.emit('signalRequest', { toUid, signal });
    }
    createCall(params) {
        this.emitter.emit('callRequest', { params });
    }
    async processCallSignal(uid, signal) {
        try {
            const call = await models_1.MediaCalls.findOneById(signal.callId);
            if (!call) {
                logger_1.logger.error({
                    msg: 'call not found',
                    method: 'GlobalSignalProcessor.processCallSignal',
                    signal: (0, stripSensitiveData_1.stripSensitiveDataFromSignal)(signal),
                });
                throw new Error('invalid-call');
            }
            const isCaller = call.caller.type === 'user' && call.caller.id === uid;
            const isCallee = call.callee.type === 'user' && call.callee.id === uid;
            // The user must be either the caller or the callee, if its none or both, we can't process it
            if (isCaller === isCallee) {
                logger_1.logger.error({
                    msg: 'failed to identify actor role in the call',
                    method: 'processSignal',
                    signal: (0, stripSensitiveData_1.stripSensitiveDataFromSignal)(signal),
                    isCaller,
                    isCallee,
                });
                throw new Error('invalid-call');
            }
            const role = isCaller ? 'caller' : 'callee';
            const callActor = call[role];
            // Hangup requests from different clients won't be coming from the signed client
            const skipContractCheck = signal.type === 'hangup' && signal.reason === 'another-client';
            // Ignore signals from different sessions if the actor is already signed
            if (!skipContractCheck && callActor.contractId && callActor.contractId !== signal.contractId) {
                return;
            }
            await CallDirector_1.mediaCallDirector.renewCallId(call._id);
            const agents = await CallDirector_1.mediaCallDirector.cast.getAgentsFromCall(call);
            const { [role]: agent } = agents;
            if (!(agent instanceof UserActorAgent_1.UserActorAgent)) {
                throw new Error('Actor agent is not prepared to process signals');
            }
            await agent.processSignal(call, signal);
        }
        catch (e) {
            logger_1.logger.error(e);
            throw e;
        }
    }
    async processRegisterSignal(uid, signal) {
        const calls = await models_1.MediaCalls.findAllNotOverByUid(uid).toArray();
        if (!calls.length) {
            return;
        }
        await Promise.all(calls.map((call) => this.reactToUnknownCall(uid, call, signal).catch(() => null)));
    }
    async reactToUnknownCall(uid, call, signal) {
        if (call.state === 'hangup') {
            return;
        }
        const isCaller = call.caller.type === 'user' && call.caller.id === uid;
        const isCallee = call.callee.type === 'user' && call.callee.id === uid;
        if (!isCaller && !isCallee) {
            return;
        }
        const role = isCaller ? 'caller' : 'callee';
        const actor = call[role];
        // If this user's side of the call has already been signed
        if (actor.contractId) {
            // If it's signed to the same session that is now registering
            // Or it was signed by a session that the current session is replacing (as in a browser refresh)
            if (actor.contractId === signal.contractId || actor.contractId === signal.oldContractId) {
                await CallDirector_1.mediaCallDirector.hangupDetachedCall(call, { endedBy: { ...actor, contractId: signal.contractId }, reason: 'unknown' });
                return;
            }
        }
        else {
            await CallDirector_1.mediaCallDirector.renewCallId(call._id);
        }
        this.sendSignal(uid, (0, buildNewCallSignal_1.buildNewCallSignal)(call, role));
        if (call.state === 'active') {
            this.sendSignal(uid, {
                callId: call._id,
                type: 'notification',
                notification: 'active',
                ...(actor.contractId && { signedContractId: actor.contractId }),
            });
        }
        else if (actor.contractId && !(0, media_signaling_1.isPendingState)(call.state)) {
            this.sendSignal(uid, {
                callId: call._id,
                type: 'notification',
                notification: 'accepted',
                signedContractId: actor.contractId,
            });
        }
    }
    async processRequestCallSignal(uid, signal) {
        logger_1.logger.debug({ msg: 'GlobalSignalProcessor.processRequestCallSignal', signal, uid });
        const existingCall = await this.getExistingRequestedCall(uid, signal);
        if (existingCall) {
            return;
        }
        const hasCalls = await models_1.MediaCalls.hasUnfinishedCallsByUid(uid);
        if (hasCalls) {
            this.rejectCallRequest(uid, { callId: signal.callId, toContractId: signal.contractId, reason: 'busy' });
        }
        const services = signal.supportedServices ?? [];
        const requestedService = services.includes('webrtc') ? 'webrtc' : services[0];
        const params = {
            caller: {
                type: 'user',
                id: uid,
                contractId: signal.contractId,
            },
            callee: signal.callee,
            requestedBy: {
                type: 'user',
                id: uid,
                contractId: signal.contractId,
            },
            requestedCallId: signal.callId,
            ...(requestedService && { requestedService }),
        };
        this.createCall(params);
    }
    async getExistingRequestedCall(uid, signal) {
        const { callId: requestedCallId } = signal;
        if (!requestedCallId) {
            return null;
        }
        logger_1.logger.debug({ msg: 'GlobalSignalProcessor.getExistingRequestedCall', uid, signal });
        const caller = { type: 'user', id: uid };
        const rejection = { callId: requestedCallId, toContractId: signal.contractId, reason: 'invalid-call-id' };
        // The requestedCallId must never match a real call id
        const matchingValidCall = await models_1.MediaCalls.findOneById(requestedCallId, { projection: { _id: 1 } });
        if (matchingValidCall) {
            this.rejectCallRequest(uid, rejection);
        }
        const call = await models_1.MediaCalls.findOneByCallerRequestedId(requestedCallId, caller);
        if (!call) {
            return null;
        }
        // if the call is already over, we treat it as an invalid id since it can't be reused
        if (call.state === 'hangup') {
            this.rejectCallRequest(uid, rejection);
        }
        if (call.caller.contractId !== signal.contractId) {
            this.rejectCallRequest(uid, { ...rejection, reason: 'existing-call-id' });
        }
        if (signal.supportedServices?.length && !signal.supportedServices.includes(call.service)) {
            this.rejectCallRequest(uid, { ...rejection, reason: 'unsupported' });
        }
        // if the call is already accepted, we won't send its signals again
        if (!(0, media_signaling_1.isPendingState)(call.state)) {
            this.rejectCallRequest(uid, { ...rejection, reason: 'already-requested' });
        }
        this.sendSignal(uid, (0, buildNewCallSignal_1.buildNewCallSignal)(call, 'caller'));
        return call;
    }
    rejectCallRequest(uid, rejection) {
        logger_1.logger.info({ msg: 'Call Request Rejected', uid, rejection });
        this.sendSignal(uid, { type: 'rejected-call-request', ...rejection });
        throw new Error(rejection.reason);
    }
}
exports.GlobalSignalProcessor = GlobalSignalProcessor;
//# sourceMappingURL=SignalProcessor.js.map