"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserActorSignalProcessor = void 0;
const media_signaling_1 = require("@rocket.chat/media-signaling");
const models_1 = require("@rocket.chat/models");
const logger_1 = require("../../logger");
const CallDirector_1 = require("../../server/CallDirector");
const injection_1 = require("../../server/injection");
const stripSensitiveData_1 = require("../../server/stripSensitiveData");
class UserActorSignalProcessor {
    get contractId() {
        return this.channel.contractId;
    }
    get callId() {
        return this.channel.callId;
    }
    get actorId() {
        return this.channel.actorId;
    }
    get actorType() {
        return this.channel.actorType;
    }
    get role() {
        return this.channel.role;
    }
    get actor() {
        return {
            type: this.actorType,
            id: this.actorId,
            contractId: this.contractId,
        };
    }
    constructor(agent, call, channel) {
        this.agent = agent;
        this.call = call;
        this.channel = channel;
        const actor = call[channel.role];
        this.signed = Boolean(actor.contractId && actor.contractId === channel.contractId);
        this.ignored = Boolean(actor.contractId && actor.contractId !== channel.contractId);
    }
    async requestWebRTCOffer(params) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.requestWebRTCOffer', actor: this.actor });
        await this.sendSignal({
            callId: this.callId,
            toContractId: this.contractId,
            type: 'request-offer',
            ...params,
        });
    }
    async processSignal(signal) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.processSignal', signal: (0, stripSensitiveData_1.stripSensitiveDataFromSignal)(signal) });
        // The code will only reach this point if one of the following conditions are true:
        // 1. the signal came from the exact user session where the caller initiated the call
        // 2. the signal came from the exact user session where the callee accepted the call
        // 3. the call has not been accepted yet and the signal came from a valid session from the callee
        // 4. It's a hangup request with reason = 'another-client' and the request came from any valid client of either user
        switch (signal.type) {
            case 'local-sdp':
                return this.saveLocalDescription(signal.sdp, signal.negotiationId);
            case 'answer':
                return this.processAnswer(signal.answer);
            case 'hangup':
                return this.hangup(signal.reason);
            case 'local-state':
                return this.reviewLocalState(signal);
            case 'error':
                return this.processError(signal);
            case 'negotiation-needed':
                return this.processNegotiationNeeded(signal.oldNegotiationId);
            case 'transfer':
                return this.processCallTransfer(signal.to);
            case 'dtmf':
                return this.processDTMF(signal.dtmf, signal.duration);
        }
    }
    async hangup(reason) {
        return CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, reason);
    }
    async saveLocalDescription(sdp, negotiationId) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.saveLocalDescription', sdp: (0, stripSensitiveData_1.stripSensitiveDataFromSdp)(sdp), signed: this.signed });
        if (!this.signed) {
            return;
        }
        await CallDirector_1.mediaCallDirector.saveWebrtcSession(this.call, this.agent, { sdp, negotiationId }, this.contractId);
    }
    async processAnswer(answer) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.processAnswer', answer });
        switch (answer) {
            case 'ack':
                return this.clientIsReachable();
            case 'accept':
                return this.clientHasAccepted();
            case 'unavailable':
                return this.clientIsUnavailable();
            case 'reject':
                return this.clientHasRejected();
        }
    }
    async processError(signal) {
        if (!this.signed) {
            return;
        }
        const { errorType = 'other', errorCode, critical = false, negotiationId, errorDetails } = signal;
        logger_1.logger.error({
            msg: 'Client reported an error',
            errorType,
            errorCode,
            critical,
            errorDetails,
            negotiationId,
            callId: this.callId,
            role: this.role,
            state: this.call.state,
        });
        let hangupReason = 'error';
        if (errorType === 'service') {
            hangupReason = 'service-error';
            // Do not hangup on service errors after the call is already active;
            // if the error happened on a renegotiation, then the service may still be able to rollback to a valid state
            if (this.isPastNegotiation()) {
                return;
            }
        }
        if (!critical) {
            return;
        }
        if (errorType === 'signaling') {
            hangupReason = 'signaling-error';
        }
        await CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, hangupReason);
    }
    async processNegotiationNeeded(oldNegotiationId) {
        // Unsigned clients may not request negotiations
        if (!this.signed) {
            return;
        }
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.processNegotiationNeeded', oldNegotiationId });
        const negotiation = await models_1.MediaCallNegotiations.findLatestByCallId(this.callId);
        // If the call doesn't even have an initial negotiation yet, the clients shouldn't be requesting new ones.
        if (!negotiation) {
            return;
        }
        // If the latest negotiation has an answer, we can accept any request
        if (negotiation.answer) {
            return this.startNewNegotiation();
        }
        const comingFromLatest = oldNegotiationId === negotiation._id;
        const isRequestImpolite = this.role === 'caller';
        const isLatestImpolite = negotiation.offerer === 'caller';
        // If the request came from a client who was not yet aware of a newer renegotiation
        if (!comingFromLatest) {
            // If the client is polite, we can ignore their request in favor of the existing renegotiation
            if (!isRequestImpolite) {
                logger_1.logger.debug({ msg: 'Ignoring outdated polite renegotiation request' });
                return;
            }
            // If the latest negotiation is impolite and the impolite client is not aware of it yet, this must be a duplicate request
            if (isLatestImpolite) {
                // If we already received an offer in this situation then something is very wrong (some proxy interfering with signals, perhaps?)
                if (negotiation.offer) {
                    logger_1.logger.error({ msg: 'Invalid renegotiation request', requestedBy: this.role, isLatestImpolite });
                    return;
                }
                // Resend the offer request to the impolite client
                return this.requestWebRTCOffer({ negotiationId: negotiation._id });
            }
            // The state of polite negotiations is irrelevant for impolite requests, so we can start a new negotiation here.
            return this.startNewNegotiation();
        }
        // The client is up-to-date and requested a renegotiation before the last one was complete
        // If the request came from the same side as the last negotiation, the client was in no position to request it
        if (this.role === negotiation.offerer) {
            logger_1.logger.error({ msg: 'Invalid state for renegotiation request', requestedBy: this.role, isLatestImpolite });
            return;
        }
        // If the request is from the impolite client, it takes priority over the existing polite negotiation
        if (isRequestImpolite) {
            return this.startNewNegotiation();
        }
        // It's a polite negotiation requested while an impolite one was not yet complete
        logger_1.logger.error({ msg: 'Invalid state for renegotiation request', requestedBy: this.role, isLatestImpolite });
    }
    async startNewNegotiation() {
        const negotiationId = await CallDirector_1.mediaCallDirector.startNewNegotiation(this.call, this.role);
        if (negotiationId) {
            await this.requestWebRTCOffer({ negotiationId });
        }
    }
    async processCallTransfer(to) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.processCallTransfer', to });
        if (!(0, media_signaling_1.isBusyState)(this.call.state)) {
            return;
        }
        const self = {
            ...this.agent.getMyCallActor(this.call),
            ...this.actor,
        };
        return CallDirector_1.mediaCallDirector.transferCall(this.call, to, self, this.agent);
    }
    async processDTMF(dtmf, duration) {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.processDTMF', dtmf, duration });
        this.agent.oppositeAgent?.onDTMF(this.call._id, dtmf, duration || 2000);
    }
    async clientIsReachable() {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.clientIsReachable', role: this.role, uid: this.actorId });
        if (this.role === 'callee' && this.call.state === 'none') {
            // Change the call state from 'none' to 'ringing' when any callee session is found
            const ringUpdateResult = await models_1.MediaCalls.startRingingById(this.callId, CallDirector_1.mediaCallDirector.getNewExpirationTime());
            if (ringUpdateResult.modifiedCount) {
                CallDirector_1.mediaCallDirector.scheduleExpirationCheckByCallId(this.callId);
            }
        }
        // The caller contract should be signed before the call even starts, so if this one isn't, ignore its state
        if (this.role === 'caller' && this.signed) {
            // When the signed caller's client is reached, we immediatelly start the first negotiation
            const negotiationId = await CallDirector_1.mediaCallDirector.startFirstNegotiation(this.call);
            if (negotiationId) {
                await this.requestWebRTCOffer({ negotiationId });
            }
        }
    }
    async clientHasRejected() {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.clientHasRejected', role: this.role, uid: this.actorId });
        if (!this.isCallPending()) {
            return;
        }
        if (this.role === 'callee') {
            return CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, 'rejected');
        }
    }
    async clientIsUnavailable() {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.clientIsUnavailable', role: this.role, uid: this.actorId });
        // Ignore 'unavailable' responses from unsigned clients as some other client session may have a different answer
        if (!this.signed) {
            return;
        }
        await CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, 'unavailable');
    }
    async clientHasAccepted() {
        logger_1.logger.debug({ msg: 'UserActorSignalProcessor.clientHasAccepted', role: this.role, uid: this.actorId });
        if (!this.isCallPending()) {
            return;
        }
        if (this.role === 'callee') {
            await CallDirector_1.mediaCallDirector.acceptCall(this.call, this.agent, { calleeContractId: this.contractId });
        }
    }
    async clientIsActive() {
        const result = await models_1.MediaCallChannels.setActiveById(this.channel._id);
        if (result.modifiedCount) {
            await CallDirector_1.mediaCallDirector.activate(this.call, this.agent);
        }
    }
    async sendSignal(signal) {
        (0, injection_1.getMediaCallServer)().sendSignal(this.actorId, signal);
    }
    isCallPending() {
        return (0, media_signaling_1.isPendingState)(this.call.state);
    }
    isPastNegotiation() {
        return ['active', 'hangup'].includes(this.call.state);
    }
    async reviewLocalState(signal) {
        if (!this.signed) {
            return;
        }
        if (signal.clientState === 'active') {
            if (signal.negotiationId) {
                void models_1.MediaCallNegotiations.setStableById(signal.negotiationId).catch(() => null);
            }
            if (this.channel.state === 'active' || this.channel.activeAt) {
                return;
            }
            await this.clientIsActive();
        }
    }
}
exports.UserActorSignalProcessor = UserActorSignalProcessor;
//# sourceMappingURL=CallSignalProcessor.js.map