"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserActorAgent = void 0;
const media_signaling_1 = require("@rocket.chat/media-signaling");
const models_1 = require("@rocket.chat/models");
const CallSignalProcessor_1 = require("./CallSignalProcessor");
const BaseAgent_1 = require("../../base/BaseAgent");
const logger_1 = require("../../logger");
const buildNewCallSignal_1 = require("../../server/buildNewCallSignal");
const injection_1 = require("../../server/injection");
class UserActorAgent extends BaseAgent_1.BaseMediaCallAgent {
    async processSignal(call, signal) {
        const channel = await this.getOrCreateChannel(call, signal.contractId);
        const signalProcessor = new CallSignalProcessor_1.UserActorSignalProcessor(this, call, channel);
        return signalProcessor.processSignal(signal);
    }
    async sendSignal(signal) {
        (0, injection_1.getMediaCallServer)().sendSignal(this.actorId, signal);
    }
    async onCallAccepted(callId, signedContractId) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onCallAccepted', callId });
        await this.sendSignal({
            callId,
            type: 'notification',
            notification: 'accepted',
            signedContractId,
        });
        if (this.role !== 'callee') {
            return;
        }
        const negotiation = await models_1.MediaCallNegotiations.findLatestByCallId(callId);
        if (!negotiation?.offer) {
            logger_1.logger.debug('The call was accepted but the webrtc offer is not yet available.');
            return;
        }
        await this.sendSignal({
            callId,
            toContractId: signedContractId,
            type: 'remote-sdp',
            sdp: negotiation.offer,
            negotiationId: negotiation._id,
        });
    }
    async onCallEnded(callId) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onCallEnded', callId });
        return this.sendSignal({
            callId,
            type: 'notification',
            notification: 'hangup',
        });
    }
    async onCallActive(callId) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onCallActive', callId });
        return this.sendSignal({
            callId,
            type: 'notification',
            notification: 'active',
        });
    }
    async onCallCreated(call) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onCallCreated', call });
        if (this.role === 'caller' && call.caller.contractId) {
            // Pre-create the channel for the contractId that requested the call
            await this.getOrCreateChannel(call, call.caller.contractId);
        }
        await this.sendSignal((0, buildNewCallSignal_1.buildNewCallSignal)(call, this.role));
    }
    async onRemoteDescriptionChanged(callId, negotiationId) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onRemoteDescriptionChanged', callId, negotiationId });
        const call = await models_1.MediaCalls.findOneById(callId);
        if (!call || !(0, media_signaling_1.isBusyState)(call.state)) {
            return;
        }
        const actor = this.getMyCallActor(call);
        const toContractId = actor.contractId;
        // Do not send any sdp to an actor until they have a signed contract
        if (!toContractId) {
            return;
        }
        const negotiation = await models_1.MediaCallNegotiations.findOneById(negotiationId);
        if (!negotiation) {
            return;
        }
        if (negotiation.offerer === this.role) {
            if (!negotiation.offer) {
                await this.sendSignal({
                    callId,
                    toContractId,
                    type: 'request-offer',
                    negotiationId,
                });
                return;
            }
            if (!negotiation.answer) {
                return;
            }
            await this.sendSignal({
                callId,
                toContractId,
                type: 'remote-sdp',
                sdp: negotiation.answer,
                negotiationId,
            });
            return;
        }
        if (!negotiation.offer) {
            return;
        }
        await this.sendSignal({
            callId,
            toContractId,
            type: 'remote-sdp',
            sdp: negotiation.offer,
            negotiationId,
        });
    }
    async onCallTransferred(callId) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onCallTransferred', callId });
        const call = await models_1.MediaCalls.findOneById(callId);
        if (!call?.transferredBy || !call?.transferredTo) {
            return;
        }
        const actor = this.getMyCallActor(call);
        // If we haven't signed yet, we can't be transferred
        if (!actor.contractId) {
            return;
        }
        await (0, injection_1.getMediaCallServer)().requestCall({
            caller: actor,
            callee: call.transferredTo,
            requestedService: call.service,
            requestedBy: call.transferredBy,
            parentCallId: call._id,
        });
    }
    async onDTMF(callId, dtmf, duration) {
        logger_1.logger.debug({ msg: 'UserActorAgent.onDTMF', callId, dtmf, duration });
        // internal calls have nothing to do with DTMFs
    }
}
exports.UserActorAgent = UserActorAgent;
//# sourceMappingURL=UserActorAgent.js.map