"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mediaCallDirector = void 0;
const models_1 = require("@rocket.chat/models");
const injection_1 = require("./injection");
const logger_1 = require("../logger");
const EXPIRATION_TIME = 120000;
const EXPIRATION_CHECK_TIMEOUT = EXPIRATION_TIME + 1000;
// expiration checks by call id
const scheduledExpirationChecks = new Map();
class MediaCallDirector {
    async hangup(call, actorAgent, reason) {
        const { actor: endedBy } = actorAgent;
        logger_1.logger.debug({ msg: 'MediaCallDirector.hangup', callId: call._id, reason, endedBy });
        const modified = await this.hangupCallById(call._id, { endedBy, reason });
        if (modified) {
            await actorAgent.onCallEnded(call._id);
            await actorAgent.oppositeAgent?.onCallEnded(call._id);
        }
    }
    async hangupByServer(call, serverErrorCode) {
        return this.hangupDetachedCall(call, { reason: serverErrorCode });
    }
    async activate(call, actorAgent) {
        logger_1.logger.debug({ msg: 'MediaCallDirector.activateCall' });
        const stateResult = await models_1.MediaCalls.activateCallById(call._id, this.getNewExpirationTime());
        if (!stateResult.modifiedCount) {
            return;
        }
        this.scheduleExpirationCheckByCallId(call._id);
        return actorAgent.oppositeAgent?.onCallActive(call._id);
    }
    async acceptCall(call, calleeAgent, data) {
        logger_1.logger.debug({ msg: 'MediaCallDirector.acceptCall' });
        // To avoid race conditions, load the negotiation before changing the call state
        // Once the state changes, negotiations need to be referred by id.
        const negotiation = await models_1.MediaCallNegotiations.findLatestByCallId(call._id);
        const { webrtcAnswer, ...acceptData } = data;
        const stateResult = await models_1.MediaCalls.acceptCallById(call._id, acceptData, this.getNewExpirationTime());
        // If nothing changed, the call was no longer ringing
        if (!stateResult.modifiedCount) {
            return false;
        }
        this.scheduleExpirationCheckByCallId(call._id);
        await calleeAgent.onCallAccepted(call._id, data.calleeContractId);
        await calleeAgent.oppositeAgent?.onCallAccepted(call._id, call.caller.contractId);
        if (data.webrtcAnswer && negotiation) {
            await models_1.MediaCallNegotiations.setAnswerById(negotiation._id, data.webrtcAnswer);
            await calleeAgent.oppositeAgent?.onRemoteDescriptionChanged(call._id, negotiation._id);
        }
        return true;
    }
    async startFirstNegotiation(call, offer) {
        const negotiation = await models_1.MediaCallNegotiations.findLatestByCallId(call._id);
        // If the call already has a negotiation, do nothing
        if (negotiation) {
            return null;
        }
        return this.startNewNegotiation(call, 'caller', offer);
    }
    async startNewNegotiation(call, offerer, offer) {
        logger_1.logger.debug({ msg: 'Adding new negotiation', callId: call._id, offerer, hasOffer: Boolean(offer) });
        const newNegotiation = {
            callId: call._id,
            offerer,
            requestTimestamp: new Date(),
            ...(offer && {
                offer,
                offerTimestamp: new Date(),
            }),
        };
        const result = await models_1.MediaCallNegotiations.insertOne(newNegotiation);
        return result.insertedId;
    }
    get cast() {
        try {
            return (0, injection_1.getCastDirector)();
        }
        catch (error) {
            logger_1.logger.error({ msg: 'Failed to access castDirector', error });
            throw error;
        }
    }
    async saveWebrtcSession(call, fromAgent, session, contractId) {
        logger_1.logger.debug({ msg: 'MediaCallDirector.saveWebrtcSession', callId: call?._id });
        const negotiation = await models_1.MediaCallNegotiations.findOneById(session.negotiationId);
        if (!negotiation) {
            throw new Error('invalid-negotiation');
        }
        const actor = fromAgent.getMyCallActor(call);
        if (!actor.contractId || actor.contractId !== contractId) {
            throw new Error('invalid-contract');
        }
        const isOfferer = fromAgent.role === negotiation.offerer;
        const isOffer = session.sdp.type === 'offer';
        if (isOffer !== isOfferer) {
            throw new Error('invalid-sdp');
        }
        const updater = isOffer
            ? models_1.MediaCallNegotiations.setOfferById(negotiation._id, session.sdp)
            : models_1.MediaCallNegotiations.setAnswerById(negotiation._id, session.sdp);
        const updateResult = await updater;
        if (!updateResult.modifiedCount) {
            return;
        }
        await fromAgent.oppositeAgent?.onRemoteDescriptionChanged(call._id, negotiation._id);
    }
    async createCall(params) {
        const { caller, callee, requestedCallId, requestedService, callerAgent, calleeAgent, parentCallId, requestedBy } = params;
        logger_1.logger.debug({
            msg: 'MediaCallDirector.createCall',
            params: { caller, callee, requestedCallId, requestedService, parentCallId, requestedBy },
        });
        // The caller must always have a contract to create the call
        if (!caller.contractId) {
            throw new Error('invalid-caller');
        }
        const service = requestedService || 'webrtc';
        // webrtc is our only known service right now, but if the call was requested by a client that doesn't also implement it, we don't need to even create a call
        if (service !== 'webrtc') {
            throw new Error('invalid-call-service');
        }
        if (!callerAgent) {
            throw new Error('invalid-caller');
        }
        if (!calleeAgent) {
            throw new Error('invalid-callee');
        }
        callerAgent.oppositeAgent = calleeAgent;
        calleeAgent.oppositeAgent = callerAgent;
        const call = {
            service,
            kind: 'direct',
            state: 'none',
            createdBy: requestedBy || caller,
            createdAt: new Date(),
            caller,
            callee,
            expiresAt: this.getNewExpirationTime(),
            uids: [
                // add actor ids to uids field if their type is 'user', to make it easy to identify any call an user was part of
                ...(caller.type === 'user' ? [caller.id] : []),
                ...(callee.type === 'user' ? [callee.id] : []),
            ],
            ended: false,
            ...(requestedCallId && { callerRequestedId: requestedCallId }),
            ...(parentCallId && { parentCallId }),
        };
        logger_1.logger.debug({ msg: 'creating call', call });
        const insertResult = await models_1.MediaCalls.insertOne(call);
        if (!insertResult.insertedId) {
            throw new Error('failed-to-create-call');
        }
        const newCall = await models_1.MediaCalls.findOneById(insertResult.insertedId);
        if (!newCall) {
            throw new Error('failed-to-retrieve-call');
        }
        this.scheduleExpirationCheckByCallId(newCall._id);
        return newCall;
    }
    async transferCall(call, to, by, agent) {
        if (!agent.oppositeAgent) {
            logger_1.logger.error('Unable to transfer calls without a reference to the opposite agent.');
            return;
        }
        const updateResult = await models_1.MediaCalls.transferCallById(call._id, { by, to });
        if (!updateResult.modifiedCount) {
            return;
        }
        await agent.oppositeAgent.onCallTransferred(call._id);
    }
    async hangupTransferredCallById(callId) {
        logger_1.logger.debug({ msg: 'MediaCallDirector.hangupTransferredCallById', callId });
        const call = await models_1.MediaCalls.findOneById(callId);
        if (!call?.transferredBy) {
            return;
        }
        return this.hangupDetachedCall(call, { endedBy: call.transferredBy, reason: 'transfer' });
    }
    async hangupExpiredCalls(expectedCallId) {
        logger_1.logger.debug('MediaCallDirector.hangupExpiredCalls');
        let expectedCallWasExpired = false;
        const result = models_1.MediaCalls.findAllExpiredCalls({
            projection: { _id: 1, caller: 1, callee: 1 },
        });
        for await (const call of result) {
            if (expectedCallId && call._id === expectedCallId) {
                expectedCallWasExpired = true;
            }
            await this.hangupByServer(call, 'timeout');
        }
        if (typeof expectedCallId === 'string') {
            return expectedCallWasExpired;
        }
    }
    getNewExpirationTime() {
        return new Date(Date.now() + EXPIRATION_TIME);
    }
    async renewCallId(callId) {
        await models_1.MediaCalls.setExpiresAtById(callId, this.getNewExpirationTime());
        this.scheduleExpirationCheckByCallId(callId);
    }
    scheduleExpirationCheckByCallId(callId) {
        const oldHandler = scheduledExpirationChecks.get(callId);
        if (oldHandler) {
            clearTimeout(oldHandler);
            scheduledExpirationChecks.delete(callId);
        }
        const handler = setTimeout(async () => {
            logger_1.logger.debug({ msg: 'MediaCallDirector.scheduleExpirationCheckByCallId.timeout', callId });
            scheduledExpirationChecks.delete(callId);
            const expectedCallWasExpired = await this.hangupExpiredCalls(callId).catch((error) => logger_1.logger.error({ msg: 'Media Call Monitor failed to hangup expired calls', error }));
            if (!expectedCallWasExpired) {
                const call = await models_1.MediaCalls.findOneById(callId, { projection: { ended: 1 } });
                if (call && !call.ended) {
                    this.scheduleExpirationCheckByCallId(callId);
                }
            }
        }, EXPIRATION_CHECK_TIMEOUT);
        scheduledExpirationChecks.set(callId, handler);
    }
    scheduleExpirationCheck() {
        setTimeout(async () => {
            logger_1.logger.debug({ msg: 'MediaCallDirector.scheduleExpirationCheck.timeout' });
            await this.hangupExpiredCalls().catch((error) => logger_1.logger.error({ msg: 'Media Call Monitor failed to hangup expired calls', error }));
        }, EXPIRATION_CHECK_TIMEOUT);
    }
    async runOnCallCreatedForAgent(call, agent, agentToNotifyIfItFails) {
        try {
            await agent.onCallCreated(call);
        }
        catch (error) {
            // If the agent failed, we assume they cleaned up after themselves and just hangup the call
            // We then notify the other agent that the call has ended, but only if it the agent was already notified about this call in the first place
            logger_1.logger.error({
                msg: 'Agent failed to process a new call.',
                error,
                agentRole: agent.role,
                callerType: call.caller.type,
                calleeType: call.callee.type,
            });
            await this.hangupCallByIdAndNotifyAgents(call._id, agentToNotifyIfItFails ? [agentToNotifyIfItFails] : [], {
                endedBy: agent.getMyCallActor(call),
                reason: 'error',
            });
            throw error;
        }
    }
    async hangupCallById(callId, params) {
        // Ensure we don't pass along anything more than the three basic actor attributes
        const endedBy = params?.endedBy?.type === 'server'
            ? { type: params.endedBy.type, id: params.endedBy.id }
            : params?.endedBy && { type: params.endedBy.type, id: params.endedBy.id, contractId: params.endedBy.contractId };
        const cleanedParams = params && {
            ...params,
            ...(endedBy && { endedBy }),
        };
        const result = await models_1.MediaCalls.hangupCallById(callId, cleanedParams).catch((error) => {
            logger_1.logger.error({
                msg: 'Failed to hangup a call.',
                callId,
                hangupError: error,
                hangupReason: params?.reason,
                hangupActor: params?.endedBy,
            });
            throw error;
        });
        const ended = Boolean(result.modifiedCount);
        if (ended) {
            (0, injection_1.getMediaCallServer)().updateCallHistory({ callId });
        }
        return ended;
    }
    async hangupCallByIdAndNotifyAgents(callId, agents, params) {
        const modified = await this.hangupCallById(callId, params).catch(() => false);
        if (!modified || !agents?.length) {
            return;
        }
        await Promise.allSettled(agents.map(async (agent) => agent.onCallEnded(callId).catch((error) => logger_1.logger.error({ msg: 'Failed to notify agent of a hangup', error, actor: agent.actor }))));
    }
    async hangupDetachedCall(call, params) {
        logger_1.logger.debug({ msg: 'MediaCallDirector.hangupDetachedCall', callId: call._id, params });
        try {
            const endedBy = params?.endedBy || { type: 'server', id: 'server' };
            const modified = await this.hangupCallById(call._id, { ...params, endedBy });
            if (!modified) {
                return;
            }
            // Try to notify the agents but there's no guarantee they are reachable
            try {
                const agents = await this.cast.getAgentsFromCall(call);
                for (const agent of Object.values(agents)) {
                    agent?.onCallEnded(call._id).catch(() => null);
                }
            }
            catch {
                // Ignore errors on the ended event
            }
        }
        catch (error) {
            logger_1.logger.error({ msg: 'Failed to terminate call.', error, callId: call._id, params });
        }
    }
}
exports.mediaCallDirector = new MediaCallDirector();
//# sourceMappingURL=CallDirector.js.map