"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IncomingSipCall = void 0;
const media_signaling_1 = require("@rocket.chat/media-signaling");
const models_1 = require("@rocket.chat/models");
const BaseSipCall_1 = require("./BaseSipCall");
const logger_1 = require("../../logger");
const BroadcastAgent_1 = require("../../server/BroadcastAgent");
const CallDirector_1 = require("../../server/CallDirector");
const injection_1 = require("../../server/injection");
const errorCodes_1 = require("../errorCodes");
class IncomingSipCall extends BaseSipCall_1.BaseSipCall {
    constructor(session, call, agent, channel, srf, req, res) {
        super(session, call, agent, channel);
        this.agent = agent;
        this.srf = srf;
        this.req = req;
        this.res = res;
        this.sipDialog = null;
        this.inboundRenegotiations = new Map();
        this.processedTransfer = false;
    }
    static async processInvite(session, srf, req, res) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.processInvite' });
        if (!req.isNewInvite) {
            logger_1.logger.error({ msg: 'IncomingSipCall.processInvite received a request that is not a new invite.' });
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.NOT_IMPLEMENTED, 'not-a-new-invite');
        }
        let sipCall = null;
        req.on('cancel', (message) => {
            sipCall?.cancel(message);
        });
        const callee = await this.getCalleeFromInvite(req);
        logger_1.logger.debug({ msg: 'incoming call to', callee });
        // getCalleeFromInvite already ensures it, but let's safeguard that the callee is an internal user
        if (callee.type !== 'user' || !callee.id) {
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
        }
        // User is literally busy
        if (await models_1.MediaCalls.hasUnfinishedCallsByUid(callee.id)) {
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
        }
        if (!(await (0, injection_1.getMediaCallServer)().permissionCheck(callee.id, 'external'))) {
            logger_1.logger.debug({ msg: 'User with no permission received a sip call.', uid: callee.id });
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
        }
        const caller = await this.getCallerContactFromInvite(session.sessionId, req);
        logger_1.logger.debug({ msg: 'incoming call from', caller });
        const webrtcOffer = { type: 'offer', sdp: req.body };
        const callerAgent = await CallDirector_1.mediaCallDirector.cast.getAgentForActorAndRole(caller, 'caller');
        if (!callerAgent) {
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.NOT_FOUND, 'Caller agent not found');
        }
        if (!(callerAgent instanceof BroadcastAgent_1.BroadcastActorAgent)) {
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.INTERNAL_SERVER_ERROR, 'Caller agent not valid');
        }
        const calleeAgent = await CallDirector_1.mediaCallDirector.cast.getAgentForActorAndRole(callee, 'callee');
        if (!calleeAgent) {
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.NOT_FOUND, 'Callee agent not found');
        }
        const call = await CallDirector_1.mediaCallDirector.createCall({
            caller,
            callee,
            callerAgent,
            calleeAgent,
        });
        const negotiationId = await CallDirector_1.mediaCallDirector.startNewNegotiation(call, 'caller', webrtcOffer);
        const channel = await callerAgent.getOrCreateChannel(call, session.sessionId);
        sipCall = new IncomingSipCall(session, call, callerAgent, channel, srf, req, res);
        callerAgent.provider = sipCall;
        sipCall.inboundRenegotiations.set(negotiationId, {
            id: negotiationId,
            req,
            res,
            isFirst: true,
            offer: webrtcOffer,
            answer: null,
        });
        // Send the call to the callee client
        await calleeAgent.onCallCreated(call);
        return sipCall;
    }
    async createDialog(localSdp) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.createDialog' });
        const uas = await this.srf.createUAS(this.req, this.res, {
            localSdp,
        });
        if (!uas) {
            logger_1.logger.debug({ msg: 'IncomingSipCall.createDialog - dialog creation failed' });
            void CallDirector_1.mediaCallDirector.hangupByServer(this.call, 'failed-to-create-sip-dialog');
            return;
        }
        uas.on('modify', async (req, res) => {
            const webrtcOffer = { type: 'offer', sdp: req.body };
            let negotiationId = null;
            logger_1.logger.debug({
                msg: 'IncomingSipCall received a renegotiation',
                callingNumber: req?.callingNumber,
                calledNumber: req?.calledNumber,
            });
            try {
                negotiationId = await CallDirector_1.mediaCallDirector.startNewNegotiation(this.call, 'caller', webrtcOffer);
                const calleeAgent = await CallDirector_1.mediaCallDirector.cast.getAgentForActorAndRole(this.call.callee, 'callee');
                if (!calleeAgent) {
                    logger_1.logger.error({ msg: 'Failed to retrieve callee agent', method: 'IncomingSipCall.uas.modify', callee: this.call.callee });
                    res.send(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
                    return;
                }
                this.inboundRenegotiations.set(negotiationId, {
                    id: negotiationId,
                    req,
                    res,
                    isFirst: false,
                    offer: webrtcOffer,
                    answer: null,
                });
                calleeAgent.onRemoteDescriptionChanged(this.call._id, negotiationId);
                logger_1.logger.debug({ msg: 'modify', method: 'IncomingSipCall.createDialog', req: this.session.stripDrachtioServerDetails(req) });
            }
            catch (error) {
                logger_1.logger.error({ msg: 'An unexpected error occured while processing a modify event on an IncomingSipCall dialog', error });
                try {
                    res.send(errorCodes_1.SipErrorCodes.INTERNAL_SERVER_ERROR);
                }
                catch {
                    //
                }
                if (!negotiationId) {
                    return;
                }
                // If we got an error after the negotiation was registered on our side, the state is unpredictable, so hangup.
                this.inboundRenegotiations.delete(negotiationId);
                this.hangupPendingCall(errorCodes_1.SipErrorCodes.INTERNAL_SERVER_ERROR);
            }
        });
        uas.on('destroy', () => {
            logger_1.logger.debug({ msg: 'IncomingSipCall - uas.destroy' });
            this.sipDialog = null;
            void CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, 'remote');
        });
        this.sipDialog = uas;
    }
    cancel(res) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.cancel', res: this.session.stripDrachtioServerDetails(res) });
        void CallDirector_1.mediaCallDirector.hangup(this.call, this.agent, 'remote').catch(() => null);
    }
    async reflectCall(call, params) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.reflectCall', call, lastCallState: this.lastCallState, params });
        if (params.dtmf && this.sipDialog) {
            return this.sendDTMF(this.sipDialog, params.dtmf.dtmf, params.dtmf.duration || 2000);
        }
        if (call.transferredTo && call.transferredBy) {
            return this.processTransferredCall(call);
        }
        if (call.ended) {
            return this.processEndedCall(call);
        }
        if ((0, media_signaling_1.isBusyState)(call.state)) {
            return this.processNegotiations(call);
        }
    }
    async processTransferredCall(call) {
        if (this.lastCallState === 'hangup' || !call.transferredTo || !call.transferredBy) {
            return;
        }
        if (!this.sipDialog || this.processedTransfer) {
            if (call.ended) {
                return this.processEndedCall(call);
            }
            return;
        }
        this.processedTransfer = true;
        try {
            // Sip targets can only be referred to other sip users
            const newCallee = await CallDirector_1.mediaCallDirector.cast.getContactForActor(call.transferredTo, { requiredType: 'sip' });
            if (!newCallee) {
                throw new Error('invalid-transfer');
            }
            const referTo = this.session.geContactUri(newCallee);
            const referredBy = this.session.geContactUri(call.transferredBy);
            const res = await this.sipDialog.request({
                method: 'REFER',
                headers: {
                    'Refer-To': referTo,
                    'Referred-By': referredBy,
                },
            });
            if (res.status === 202) {
                logger_1.logger.debug({ msg: 'REFER was accepted', method: 'IncomingSipCall.processTransferredCall' });
            }
        }
        catch (error) {
            logger_1.logger.debug({ msg: 'REFER failed', method: 'IncomingSipCall.processTransferredCall', error });
            if (!call.ended) {
                void CallDirector_1.mediaCallDirector.hangupByServer(call, 'sip-refer-failed');
            }
            return this.processEndedCall(call);
        }
    }
    async processEndedCall(call) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.processEndedCall' });
        switch (call.hangupReason) {
            case 'service-error':
                this.cancelPendingInvites(errorCodes_1.SipErrorCodes.NOT_ACCEPTABLE_HERE);
                break;
            case 'rejected':
                this.cancelPendingInvites(errorCodes_1.SipErrorCodes.DECLINED);
                break;
            default:
                this.cancelPendingInvites(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
                break;
        }
        if (this.lastCallState === 'hangup') {
            return;
        }
        const { sipDialog } = this;
        this.sipDialog = null;
        this.lastCallState = 'hangup';
        if (sipDialog) {
            sipDialog.destroy();
        }
    }
    async getPendingInboundNegotiation() {
        for await (const localNegotiation of this.inboundRenegotiations.values()) {
            if (localNegotiation.answer) {
                continue;
            }
            // If the negotiation does not exist, remove it from the list
            const negotiation = await models_1.MediaCallNegotiations.findOneById(localNegotiation.id);
            // Negotiation will always exist; This is just a safe guard
            if (!negotiation) {
                logger_1.logger.error({ msg: 'Invalid Negotiation reference on IncomingSipCall.', localNegotiation: localNegotiation.id });
                this.inboundRenegotiations.delete(localNegotiation.id);
                if (localNegotiation.res) {
                    localNegotiation.res.send(errorCodes_1.SipErrorCodes.INTERNAL_SERVER_ERROR);
                }
                continue;
            }
            if (negotiation.answer) {
                localNegotiation.answer = negotiation.answer;
            }
            return localNegotiation;
        }
        return null;
    }
    async processNegotiations(call) {
        const localNegotiation = await this.getPendingInboundNegotiation();
        if (!localNegotiation) {
            // Callee-initiated renegotiations are only processed if there's none initiated by the caller
            return this.processCalleeNegotiation(call);
        }
        // If we don't have an sdp, we can't respond to it yet
        if (!localNegotiation?.answer?.sdp) {
            return;
        }
        if (localNegotiation.isFirst) {
            return this.createDialog(localNegotiation.answer.sdp).catch(() => {
                logger_1.logger.error('Failed to create incoming call dialog.');
                this.hangupPendingCall(errorCodes_1.SipErrorCodes.INTERNAL_SERVER_ERROR);
            });
        }
        localNegotiation.res.send(200, {
            body: localNegotiation.answer.sdp,
        });
    }
    async processCalleeNegotiation(call) {
        if (!this.sipDialog) {
            return;
        }
        const negotiation = await models_1.MediaCallNegotiations.findLatestByCallId(call._id);
        if (negotiation?.offerer !== 'callee' || !negotiation.offer?.sdp || negotiation.answer) {
            return;
        }
        logger_1.logger.debug('IncomingSipCall.processCalleeNegotiation');
        let answer = undefined;
        try {
            answer = await this.sipDialog.modify(negotiation.offer.sdp).catch(() => {
                logger_1.logger.debug('modify failed');
            });
        }
        catch (error) {
            logger_1.logger.error({ msg: 'Error on IncomingSipCall.processCalleeNegotiation', error });
        }
        if (!answer) {
            logger_1.logger.error({ msg: 'No answer from callee initiated negotiation' });
            return;
        }
        await CallDirector_1.mediaCallDirector.saveWebrtcSession(call, this.agent, { sdp: { sdp: answer, type: 'answer' }, negotiationId: negotiation._id }, this.session.sessionId);
    }
    cancelPendingInvites(errorCode) {
        logger_1.logger.debug({
            msg: 'IncomingSipCall.cancelPendingInvites',
            errorCode,
            hasDialog: Boolean(this.sipDialog),
            negotiations: this.inboundRenegotiations.size,
        });
        for (const localNegotiation of this.inboundRenegotiations.values()) {
            // if it has an answer sdp, we already responded to it
            if (localNegotiation.answer?.sdp) {
                continue;
            }
            try {
                localNegotiation.res.send(errorCode);
            }
            catch {
                //
            }
        }
        this.inboundRenegotiations.clear();
    }
    hangupPendingCall(errorCode) {
        logger_1.logger.debug('IncomingSipCall.hangupPendingCall');
        this.cancelPendingInvites(errorCode);
        void CallDirector_1.mediaCallDirector.hangupByServer(this.call, `sip-error-${errorCode}`);
    }
    static async getCalleeFromInvite(req) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.getCalleeFromInvite', callingNumber: req.callingNumber, calledNumber: req.calledNumber });
        if (req.calledNumber && typeof req.calledNumber === 'string') {
            const userContact = await CallDirector_1.mediaCallDirector.cast.getContactForExtensionNumber(req.calledNumber, { requiredType: 'user' });
            if (userContact) {
                return userContact;
            }
            // If the invite had an id/extension but we couldn't match it to an user, respond with unavailable
            throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.TEMPORARILY_UNAVAILABLE);
        }
        // If we couldn't even identify an id/extension from the invite, respond with not found
        throw new errorCodes_1.SipError(errorCodes_1.SipErrorCodes.NOT_FOUND);
    }
    static async getRocketChatCallerFromInvite(req) {
        logger_1.logger.debug({
            msg: 'IncomingSipCall.getRocketChatCallerFromInvite',
            callingNumber: req.callingNumber,
            calledNumber: req.calledNumber,
        });
        if (req.callingNumber && typeof req.callingNumber === 'string') {
            const userContact = await CallDirector_1.mediaCallDirector.cast.getContactForExtensionNumber(req.callingNumber, { preferredType: 'sip' });
            if (userContact) {
                return userContact;
            }
        }
        return null;
    }
    static async getCallerContactFromInvite(sessionId, req) {
        logger_1.logger.debug({ msg: 'IncomingSipCall.getCallerContactFromInvite' });
        const callerBase = await this.getRocketChatCallerFromInvite(req);
        const displayNameFromHeader = req.has('X-RocketChat-Caller-Name') && req.get('X-RocketChat-Caller-Name');
        const usernameFromHeader = req.has('X-RocketChat-Caller-Username') && req.get('X-RocketChat-Caller-Username');
        const displayName = displayNameFromHeader || callerBase?.displayName || req.from;
        const username = usernameFromHeader || callerBase?.username || req.callingNumber;
        const sipExtension = req.callingNumber;
        const defaultContactInfo = {
            username,
            sipExtension,
            displayName: displayName || sipExtension,
        };
        const contact = await CallDirector_1.mediaCallDirector.cast.getContactForExtensionNumber(sipExtension, { requiredType: 'sip' }, defaultContactInfo);
        if (contact) {
            return {
                ...contact,
                contractId: sessionId,
            };
        }
        return {
            type: 'sip',
            id: sipExtension,
            contractId: sessionId,
            ...defaultContactInfo,
        };
    }
}
exports.IncomingSipCall = IncomingSipCall;
//# sourceMappingURL=IncomingSipCall.js.map