"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseRaw = void 0;
const tracing_1 = require("@rocket.chat/tracing");
const mongodb_1 = require("mongodb");
const __1 = require("..");
const setUpdatedAt_1 = require("./setUpdatedAt");
const warnFields = process.env.NODE_ENV !== 'production' || process.env.SHOW_WARNINGS === 'true'
    ? (...rest) => {
        console.warn(...rest, new Error().stack);
    }
    : new Function();
class BaseRaw {
    /**
     * @param db MongoDB instance
     * @param name Name of the model without any prefix. Used by trash records to set the `__collection__` field.
     * @param trash Trash collection instance
     * @param options Model options
     */
    constructor(db, name, trash, options) {
        this.db = db;
        this.name = name;
        this.trash = trash;
        this.collectionName = options?.collectionNameResolver ? options.collectionNameResolver(name) : (0, __1.getCollectionName)(name);
        this.col = this.db.collection(this.collectionName, options?.collection || {});
        void this.createIndexes();
        this.preventSetUpdatedAt = options?.preventSetUpdatedAt ?? false;
        return (0, tracing_1.traceInstanceMethods)(this);
    }
    async createIndexes() {
        const indexes = this.modelIndexes();
        if (indexes?.length) {
            if (this.pendingIndexes) {
                await this.pendingIndexes;
            }
            this.pendingIndexes = this.col.createIndexes(indexes).catch((e) => {
                console.warn(`Some indexes for collection '${this.collectionName}' could not be created:\n\t${e.message}`);
            });
            void this.pendingIndexes.finally(() => {
                this.pendingIndexes = undefined;
            });
            return this.pendingIndexes;
        }
    }
    modelIndexes() {
        return undefined;
    }
    getCollectionName() {
        return this.collectionName;
    }
    getUpdater() {
        return new __1.UpdaterImpl();
    }
    updateFromUpdater(query, updater, options = {}) {
        const updateFilter = updater.getUpdateFilter();
        return this.updateOne(query, updateFilter, options).catch((e) => {
            console.warn(e, updateFilter);
            return Promise.reject(e);
        });
    }
    doNotMixInclusionAndExclusionFields(options = {}) {
        const optionsDef = this.ensureDefaultFields(options);
        if (optionsDef?.projection === undefined) {
            return optionsDef;
        }
        const projection = optionsDef?.projection;
        const keys = Object.keys(projection);
        const removeKeys = keys.filter((key) => projection[key] === 0);
        if (keys.length > removeKeys.length) {
            removeKeys.forEach((key) => delete projection[key]);
        }
        return {
            ...optionsDef,
            projection,
        };
    }
    ensureDefaultFields(options) {
        if (options?.fields) {
            warnFields("Using 'fields' in models is deprecated.", options);
        }
        if (this.defaultFields === undefined) {
            return options;
        }
        const { fields: deprecatedFields, projection, ...rest } = options || {};
        const fields = { ...deprecatedFields, ...projection };
        return {
            projection: this.defaultFields,
            ...(fields && Object.values(fields).length && { projection: fields }),
            ...rest,
        };
    }
    findOneAndUpdate(query, update, options) {
        this.setUpdatedAt(update);
        if (options?.upsert && !('_id' in update || (update.$set && '_id' in update.$set)) && !('_id' in query)) {
            update.$setOnInsert = {
                ...(update.$setOnInsert || {}),
                _id: new mongodb_1.ObjectId().toHexString(),
            };
        }
        return this.col.findOneAndUpdate(query, update, options || {});
    }
    async findOneById(_id, options) {
        const query = { _id };
        if (options) {
            return this.findOne(query, options);
        }
        return this.findOne(query);
    }
    async findOne(query = {}, options) {
        const q = typeof query === 'string' ? { _id: query } : query;
        const optionsDef = this.doNotMixInclusionAndExclusionFields(options);
        if (optionsDef) {
            return this.col.findOne(q, optionsDef);
        }
        return this.col.findOne(q);
    }
    find(query = {}, options) {
        const optionsDef = this.doNotMixInclusionAndExclusionFields(options);
        return this.col.find(query, optionsDef);
    }
    findPaginated(query = {}, options) {
        const optionsDef = this.doNotMixInclusionAndExclusionFields(options);
        const cursor = optionsDef ? this.col.find(query, optionsDef) : this.col.find(query);
        const totalCount = this.col.countDocuments(query);
        return {
            cursor,
            totalCount,
        };
    }
    /**
     * @deprecated use {@link updateOne} or {@link updateAny} instead
     */
    update(filter, update, options) {
        const operation = options?.multi ? 'updateMany' : 'updateOne';
        return this[operation](filter, update, options);
    }
    updateOne(filter, update, options) {
        this.setUpdatedAt(update);
        if (options) {
            if (options.upsert && !('_id' in update || (update.$set && '_id' in update.$set)) && !('_id' in filter)) {
                update.$setOnInsert = {
                    ...(update.$setOnInsert || {}),
                    _id: new mongodb_1.ObjectId().toHexString(),
                };
            }
            return this.col.updateOne(filter, update, options);
        }
        return this.col.updateOne(filter, update);
    }
    updateMany(filter, update, options) {
        this.setUpdatedAt(update);
        if (options) {
            return this.col.updateMany(filter, update, options);
        }
        return this.col.updateMany(filter, update);
    }
    insertMany(docs, options) {
        docs = docs.map((doc) => {
            if (!doc._id || typeof doc._id !== 'string') {
                const oid = new mongodb_1.ObjectId();
                return { _id: oid.toHexString(), ...doc };
            }
            this.setUpdatedAt(doc);
            return doc;
        });
        // TODO reavaluate following type casting
        return this.col.insertMany(docs, options || {});
    }
    insertOne(doc, options) {
        if (!doc._id || typeof doc._id !== 'string') {
            const oid = new mongodb_1.ObjectId();
            doc = { _id: oid.toHexString(), ...doc };
        }
        this.setUpdatedAt(doc);
        // TODO reavaluate following type casting
        return this.col.insertOne(doc, options || {});
    }
    removeById(_id, options) {
        return this.deleteOne({ _id }, { session: options?.session });
    }
    removeByIds(ids) {
        return this.deleteMany({ _id: { $in: ids } });
    }
    async deleteOne(filter, options) {
        if (!this.trash) {
            if (options) {
                return this.col.deleteOne(filter, options);
            }
            return this.col.deleteOne(filter);
        }
        const doc = await this.findOne(filter);
        if (doc) {
            const { _id, ...record } = doc;
            const trash = {
                ...record,
                _deletedAt: new Date(),
                __collection__: this.name,
            };
            // since the operation is not atomic, we need to make sure that the record is not already deleted/inserted
            await this.trash?.updateOne({ _id }, { $set: trash }, {
                upsert: true,
            });
        }
        if (options) {
            return this.col.deleteOne(filter, options);
        }
        return this.col.deleteOne(filter);
    }
    async findOneAndDelete(filter, options) {
        if (!this.trash) {
            return this.col.findOneAndDelete(filter, options || {});
        }
        const doc = await this.col.findOne(filter);
        if (!doc) {
            return null;
        }
        const { _id, ...record } = doc;
        const trash = {
            ...record,
            _deletedAt: new Date(),
            __collection__: this.name,
        };
        await this.trash?.updateOne({ _id }, { $set: trash }, {
            upsert: true,
        });
        try {
            await this.col.deleteOne({ _id });
        }
        catch (e) {
            await this.trash?.deleteOne({ _id });
            throw e;
        }
        return doc;
    }
    async deleteMany(filter, options) {
        if (!this.trash) {
            if (options) {
                return this.col.deleteMany(filter, options);
            }
            return this.col.deleteMany(filter);
        }
        const cursor = this.find(filter, { session: options?.session });
        const ids = [];
        for await (const doc of cursor) {
            const { _id, ...record } = doc;
            const trash = {
                ...record,
                _deletedAt: new Date(),
                __collection__: this.name,
            };
            ids.push(_id);
            // since the operation is not atomic, we need to make sure that the record is not already deleted/inserted
            await this.trash?.updateOne({ _id }, { $set: trash }, {
                upsert: true,
                session: options?.session,
            });
            void options?.onTrash?.(doc);
        }
        if (options) {
            return this.col.deleteMany({ _id: { $in: ids } }, options);
        }
        return this.col.deleteMany({ _id: { $in: ids } });
    }
    // Trash
    trashFind(query, options) {
        if (!this.trash) {
            return undefined;
        }
        if (options) {
            return this.trash.find({
                __collection__: this.name,
                ...query,
            }, options);
        }
        return this.trash.find({
            __collection__: this.name,
            ...query,
        });
    }
    async trashFindOneById(_id, options) {
        const query = {
            _id,
            __collection__: this.name,
        };
        if (!this.trash) {
            return null;
        }
        if (options) {
            return this.trash.findOne(query, options);
        }
        return this.trash.findOne(query);
    }
    setUpdatedAt(record) {
        if (this.preventSetUpdatedAt) {
            return;
        }
        (0, setUpdatedAt_1.setUpdatedAt)(record);
    }
    trashFindDeletedAfter(deletedAt, query, options) {
        const q = {
            __collection__: this.name,
            _deletedAt: {
                $gt: deletedAt,
            },
            ...query,
        };
        if (!this.trash) {
            throw new Error('Trash is not enabled for this collection');
        }
        if (options) {
            return this.trash.find(q, options);
        }
        return this.trash.find(q);
    }
    trashFindPaginatedDeletedAfter(deletedAt, query, options) {
        const q = {
            __collection__: this.name,
            _deletedAt: {
                $gt: deletedAt,
            },
            ...query,
        };
        if (!this.trash) {
            throw new Error('Trash is not enabled for this collection');
        }
        const cursor = options ? this.trash.find(q, options) : this.trash.find(q);
        const totalCount = this.trash.countDocuments(q);
        return {
            cursor,
            totalCount,
        };
    }
    watch(pipeline) {
        return this.col.watch(pipeline);
    }
    countDocuments(query, options) {
        if (options) {
            return this.col.countDocuments(query, options);
        }
        return this.col.countDocuments(query);
    }
    estimatedDocumentCount() {
        return this.col.estimatedDocumentCount();
    }
}
exports.BaseRaw = BaseRaw;
//# sourceMappingURL=BaseRaw.js.map