"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CalendarEventRaw = void 0;
const BaseRaw_1 = require("./BaseRaw");
class CalendarEventRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'calendar_event', trash);
    }
    modelIndexes() {
        return [
            {
                key: { startTime: -1, uid: 1, externalId: 1 },
            },
            {
                key: { reminderTime: -1, notificationSent: 1 },
            },
        ];
    }
    async findOneByExternalIdAndUserId(externalId, uid) {
        return this.findOne({
            externalId,
            uid,
        });
    }
    findByUserIdAndDate(uid, date) {
        const startTime = new Date(date.toISOString());
        startTime.setHours(0, 0, 0, 0);
        const finalTime = new Date(date.valueOf());
        finalTime.setDate(finalTime.getDate() + 1);
        return this.find({
            uid,
            startTime: { $gte: startTime, $lt: finalTime },
        }, {
            sort: { startTime: 1 },
        });
    }
    async updateEvent(eventId, { subject, description, startTime, endTime, meetingUrl, reminderMinutesBeforeStart, reminderTime, previousStatus, busy, }) {
        return this.updateOne({ _id: eventId }, {
            $set: {
                ...(subject !== undefined ? { subject } : {}),
                ...(description !== undefined ? { description } : {}),
                ...(startTime ? { startTime } : {}),
                ...(endTime && { endTime }),
                ...(meetingUrl !== undefined ? { meetingUrl } : {}),
                ...(reminderMinutesBeforeStart ? { reminderMinutesBeforeStart } : {}),
                ...(reminderTime ? { reminderTime } : {}),
                ...(previousStatus ? { previousStatus } : {}),
                ...(typeof busy === 'boolean' && { busy }),
            },
        });
    }
    async findNextNotificationDate() {
        const nextEvent = await this.findOne({
            reminderTime: {
                $gt: new Date(),
            },
            notificationSent: false,
        }, {
            sort: {
                reminderTime: 1,
            },
            projection: {
                reminderTime: 1,
            },
        });
        return nextEvent?.reminderTime || null;
    }
    findEventsToNotify(notificationTime, minutes) {
        // Find all the events between notificationTime and +minutes that have not been notified yet
        const maxDate = new Date(notificationTime.toISOString());
        maxDate.setMinutes(maxDate.getMinutes() + minutes);
        return this.find({
            reminderTime: {
                $gte: notificationTime,
                $lt: maxDate,
            },
            notificationSent: false,
        }, {
            sort: {
                reminderTime: 1,
            },
        });
    }
    async flagNotificationSent(eventId) {
        return this.updateOne({
            _id: eventId,
        }, {
            $set: {
                notificationSent: true,
            },
        });
    }
    findOverlappingEvents(eventId, uid, startTime, endTime) {
        return this.find({
            _id: { $ne: eventId }, // Exclude current event
            uid,
            busy: { $ne: false },
            $or: [
                // Event starts during our event
                { startTime: { $gte: startTime, $lt: endTime } },
                // Event ends during our event
                { endTime: { $gt: startTime, $lte: endTime } },
                // Event completely contains our event
                { startTime: { $lte: startTime }, endTime: { $gte: endTime } },
            ],
        });
    }
    findEligibleEventsForCancelation(uid, endTime) {
        return this.find({
            uid,
            startTime: { $exists: true, $lte: endTime },
            endTime: { $exists: true, $gte: endTime },
        });
    }
    findEventsToScheduleNow(now, endTime) {
        return this.find({
            startTime: { $gte: now, $lt: endTime },
            busy: { $ne: false },
            endTime: { $exists: true },
        }, {
            sort: { startTime: 1 },
            projection: {
                _id: 1,
                uid: 1,
                startTime: 1,
                endTime: 1,
            },
        });
    }
    async findNextFutureEvent(startTime) {
        return this.findOne({
            startTime: { $gte: startTime },
            busy: { $ne: false },
            endTime: { $exists: true },
        }, {
            sort: { startTime: 1 },
            projection: {
                startTime: 1,
            },
        });
    }
    findEventsStartingNow({ now, offset = 1000 }) {
        return this.find({
            startTime: {
                $gte: new Date(now.getTime() - offset),
                $lt: new Date(now.getTime() + offset),
            },
            busy: { $ne: false },
        }, {
            projection: {
                _id: 1,
                uid: 1,
                startTime: 1,
                endTime: 1,
            },
        });
    }
    findEventsEndingNow({ now, offset = 1000 }) {
        return this.find({
            endTime: {
                $gte: new Date(now.getTime() - offset),
                $lt: new Date(now.getTime() + offset),
            },
            busy: { $ne: false },
        }, {
            projection: {
                _id: 1,
                uid: 1,
                startTime: 1,
                endTime: 1,
                previousStatus: 1,
            },
        });
    }
    findInProgressEvents(now) {
        return this.find({
            startTime: { $lt: now },
            endTime: { $gt: now },
            busy: { $ne: false },
        }, {
            projection: {
                _id: 1,
                uid: 1,
                startTime: 1,
                endTime: 1,
            },
        });
    }
}
exports.CalendarEventRaw = CalendarEventRaw;
//# sourceMappingURL=CalendarEvent.js.map