"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LivechatContactsRaw = void 0;
const string_helpers_1 = require("@rocket.chat/string-helpers");
const BaseRaw_1 = require("./BaseRaw");
const readSecondaryPreferred_1 = require("../readSecondaryPreferred");
class LivechatContactsRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'livechat_contact', trash);
    }
    modelIndexes() {
        return [
            {
                key: { name: 1 },
                unique: false,
                name: 'name_insensitive',
                collation: { locale: 'en', strength: 2, caseLevel: false },
            },
            {
                key: { 'emails.address': 1 },
                unique: false,
                name: 'emails_insensitive',
                partialFilterExpression: { emails: { $exists: true } },
                collation: { locale: 'en', strength: 2, caseLevel: false },
            },
            {
                key: { 'phones.phoneNumber': 1 },
                partialFilterExpression: { phones: { $exists: true } },
                unique: false,
            },
            {
                key: {
                    'channels.visitor.visitorId': 1,
                    'channels.visitor.source.type': 1,
                    'channels.visitor.source.id': 1,
                },
                name: 'visitorAssociation',
                unique: false,
            },
            {
                key: {
                    'channels.field': 1,
                    'channels.value': 1,
                    'channels.verified': 1,
                },
                partialFilterExpression: { 'channels.verified': true },
                name: 'verificationKey',
                unique: false,
            },
            {
                key: {
                    preRegistration: 1,
                },
                sparse: true,
                unique: false,
            },
            {
                key: { activity: 1 },
                sparse: true,
            },
            {
                key: { channels: 1 },
                unique: false,
            },
            {
                key: { 'channels.blocked': 1 },
                sparse: true,
            },
            {
                key: { 'channels.verified': 1 },
                sparse: true,
            },
            {
                key: { unknown: 1 },
                unique: false,
            },
        ];
    }
    async insertContact(data) {
        const result = await this.insertOne({
            createdAt: new Date(),
            ...data,
            preRegistration: !data.channels.length,
        });
        return result.insertedId;
    }
    async updateContact(contactId, data, options) {
        const updatedValue = await this.findOneAndUpdate({ _id: contactId, enabled: { $ne: false } }, { $set: { ...data, unknown: false, ...(data.channels && { preRegistration: !data.channels.length }) } }, { returnDocument: 'after', ...options });
        return updatedValue;
    }
    updateById(contactId, update, options) {
        return this.updateOne({ _id: contactId, enabled: { $ne: false } }, update, options);
    }
    async updateContactCustomFields(contactId, dataToUpdate, options) {
        if (!dataToUpdate.customFields && !dataToUpdate.conflictingFields) {
            throw new Error('At least one of customFields or conflictingFields must be provided');
        }
        return this.findOneAndUpdate({ _id: contactId, enabled: { $ne: false } }, {
            $set: { ...dataToUpdate },
        }, { returnDocument: 'after', ...options });
    }
    findPaginatedContacts(search, options) {
        const { searchText, unknown = false } = search;
        const searchRegex = (0, string_helpers_1.escapeRegExp)(searchText || '');
        const match = {
            $or: [
                { name: { $regex: searchRegex, $options: 'i' } },
                { 'emails.address': { $regex: searchRegex, $options: 'i' } },
                { 'phones.phoneNumber': { $regex: searchRegex, $options: 'i' } },
            ],
            unknown,
            enabled: { $ne: false },
        };
        return this.findPaginated({ ...match }, {
            allowDiskUse: true,
            ...options,
        });
    }
    async findContactMatchingVisitor(visitor) {
        const emails = visitor.visitorEmails?.map(({ address }) => address).filter((email) => Boolean(email)) || [];
        const phoneNumbers = visitor.phone?.map(({ phoneNumber }) => phoneNumber).filter((phone) => Boolean(phone)) || [];
        if (!emails?.length && !phoneNumbers?.length) {
            return null;
        }
        const query = {
            $and: [
                {
                    $or: [
                        ...emails?.map((email) => ({ 'emails.address': email })),
                        ...phoneNumbers?.map((phone) => ({ 'phones.phoneNumber': phone })),
                    ],
                },
                {
                    preRegistration: true,
                },
            ],
        };
        return this.findOne(query);
    }
    async findContactByEmailAndContactManager(email) {
        return this.findOne({ emails: { $elemMatch: { address: email } }, contactManager: { $exists: true } }, { projection: { contactManager: 1 } });
    }
    makeQueryForVisitor(visitor, extraFilters) {
        return {
            channels: {
                $elemMatch: {
                    'visitor.visitorId': visitor.visitorId,
                    'visitor.source.type': visitor.source.type,
                    ...(visitor.source.id ? { 'visitor.source.id': visitor.source.id } : {}),
                    ...extraFilters,
                },
            },
        };
    }
    async findOneByVisitor(visitor, options = {}) {
        return this.findOne(this.makeQueryForVisitor(visitor), options);
    }
    async addChannel(contactId, channel) {
        await this.updateOne({ _id: contactId, enabled: { $ne: false } }, { $push: { channels: channel }, $set: { preRegistration: false } });
    }
    async updateLastChatById(contactId, visitor, lastChat) {
        return this.updateOne({
            ...this.makeQueryForVisitor(visitor),
            _id: contactId,
        }, { $set: { lastChat, 'channels.$.lastChat': lastChat } });
    }
    async isChannelBlocked(visitor) {
        return Boolean(await this.findOne(this.makeQueryForVisitor(visitor, { blocked: true }), { projection: { _id: 1 } }));
    }
    setChannelBlockStatus(visitor, blocked) {
        return this.updateOne(this.makeQueryForVisitor(visitor), { $set: { 'channels.$.blocked': blocked } });
    }
    setChannelVerifiedStatus(visitor, verified) {
        return this.updateOne(this.makeQueryForVisitor(visitor), {
            $set: {
                'channels.$.verified': verified,
                ...(verified && { 'channels.$.verifiedAt': new Date() }),
            },
        });
    }
    setVerifiedUpdateQuery(verified, contactUpdater) {
        if (verified) {
            contactUpdater.set('channels.$.verifiedAt', new Date());
        }
        return contactUpdater.set('channels.$.verified', verified);
    }
    setFieldAndValueUpdateQuery(field, value, contactUpdater) {
        contactUpdater.set('channels.$.field', field);
        return contactUpdater.set('channels.$.value', value);
    }
    updateFromUpdaterByAssociation(visitor, contactUpdater, options = {}) {
        return this.updateFromUpdater(this.makeQueryForVisitor(visitor), contactUpdater, options);
    }
    async findSimilarVerifiedContacts({ field, value }, originalContactId, options) {
        return this.find({
            channels: {
                $elemMatch: {
                    field,
                    value,
                    verified: true,
                },
            },
            _id: { $ne: originalContactId },
        }, options).toArray();
    }
    findAllByVisitorId(visitorId) {
        return this.find({
            'channels.visitor.visitorId': visitorId,
        });
    }
    async findOneEnabledById(_id, options) {
        return this.findOne({ _id, enabled: { $ne: false } }, options);
    }
    disableByVisitorId(visitorId) {
        return this.updateOne({ 'channels.visitor.visitorId': visitorId }, {
            $set: { enabled: false },
            $unset: {
                emails: 1,
                customFields: 1,
                lastChat: 1,
                channels: 1,
                name: 1,
                phones: 1,
            },
        });
    }
    disableByContactId(contactId) {
        return this.updateOne({ _id: contactId }, {
            $set: { enabled: false },
            $unset: {
                emails: 1,
                customFields: 1,
                lastChat: 1,
                channels: 1,
                name: 1,
                phones: 1,
                conflictingFields: 1,
            },
        });
    }
    async addEmail(contactId, email) {
        const updatedContact = await this.findOneAndUpdate({ _id: contactId }, { $addToSet: { emails: { address: email } } });
        return updatedContact;
    }
    isContactActiveOnPeriod(visitor, period) {
        const query = {
            ...this.makeQueryForVisitor(visitor),
            activity: period,
        };
        return this.countDocuments(query);
    }
    markContactActiveForPeriod(visitor, period) {
        const update = {
            $push: {
                activity: {
                    $each: [period],
                    $slice: -12,
                },
            },
        };
        return this.updateOne(this.makeQueryForVisitor(visitor), update);
    }
    countContactsOnPeriod(period) {
        return this.countDocuments({
            activity: period,
        });
    }
    countByContactInfo({ contactId, email, phone }) {
        const filter = {
            ...(email && { 'emails.address': email }),
            ...(phone && { 'phones.phoneNumber': phone }),
            ...(contactId && { _id: contactId }),
        };
        return this.countDocuments(filter);
    }
    countUnknown() {
        return this.countDocuments({ unknown: true }, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    countBlocked() {
        return this.countDocuments({ 'channels.blocked': true }, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    countFullyBlocked() {
        return this.countDocuments({
            'channels.blocked': true,
            'channels': { $not: { $elemMatch: { $or: [{ blocked: false }, { blocked: { $exists: false } }] } } },
        }, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    countVerified() {
        return this.countDocuments({ 'channels.verified': true }, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    countContactsWithoutChannels() {
        return this.countDocuments({ channels: { $size: 0 } }, { readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    getStatistics() {
        return this.col.aggregate([
            {
                $group: {
                    _id: null,
                    totalConflicts: {
                        $sum: { $size: { $cond: [{ $isArray: '$conflictingFields' }, '$conflictingFields', []] } },
                    },
                    avgChannelsPerContact: {
                        $avg: { $size: { $cond: [{ $isArray: '$channels' }, '$channels', []] } },
                    },
                },
            },
        ], { allowDiskUse: true, readPreference: (0, readSecondaryPreferred_1.readSecondaryPreferred)() });
    }
    updateByVisitorId(visitorId, update, options) {
        return this.updateOne({ 'channels.visitor.visitorId': visitorId }, update, options);
    }
}
exports.LivechatContactsRaw = LivechatContactsRaw;
//# sourceMappingURL=LivechatContacts.js.map