"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LivechatVisitorsRaw = void 0;
const string_helpers_1 = require("@rocket.chat/string-helpers");
const mongodb_1 = require("mongodb");
const index_1 = require("../index");
const BaseRaw_1 = require("./BaseRaw");
class LivechatVisitorsRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'livechat_visitor', trash);
    }
    modelIndexes() {
        return [
            { key: { token: 1 } },
            { key: { 'phone.phoneNumber': 1 }, sparse: true },
            { key: { 'visitorEmails.address': 1 }, sparse: true },
            { key: { name: 1 }, sparse: true },
            { key: { username: 1 } },
            { key: { 'contactMananger.username': 1 }, sparse: true },
            { key: { 'livechatData.$**': 1 } },
            // TODO: remove this index in the next major release (8.0.0)
            // { key: { activity: 1 }, partialFilterExpression: { activity: { $exists: true } } },
            { key: { disabled: 1 }, partialFilterExpression: { disabled: { $exists: true } } },
        ];
    }
    findOneVisitorByPhone(phone) {
        const query = {
            'phone.phoneNumber': phone,
        };
        return this.findOne(query);
    }
    async findOneGuestByEmailAddress(emailAddress) {
        if (!emailAddress) {
            return null;
        }
        const query = {
            'visitorEmails.address': String(emailAddress).toLowerCase(),
        };
        return this.findOne(query);
    }
    /**
     * Find visitors by _id
     * @param {string} token - Visitor token
     */
    findById(_id, options) {
        const query = {
            _id,
        };
        return this.find(query, options);
    }
    findEnabled(query, options) {
        return this.find({
            ...query,
            disabled: { $ne: true },
        }, options);
    }
    findOneEnabledById(_id, options) {
        const query = {
            _id,
            disabled: { $ne: true },
        };
        return this.findOne(query, options);
    }
    findVisitorByToken(token) {
        const query = {
            token,
            disabled: { $ne: true },
        };
        return this.find(query);
    }
    getVisitorByToken(token, options) {
        const query = {
            token,
        };
        return this.findOne(query, options);
    }
    countVisitorsBetweenDate({ start, end, department }) {
        const query = {
            disabled: { $ne: true },
            _updatedAt: {
                $gte: new Date(start),
                $lt: new Date(end),
            },
            ...(department && department !== 'undefined' && { department }),
        };
        return this.countDocuments(query);
    }
    async getNextVisitorUsername() {
        // TODO remove dependency from another model - this logic should be inside a service/function
        const livechatCount = await index_1.Settings.incrementValueById('Livechat_guest_count', 1, { returnDocument: 'after' });
        if (!livechatCount) {
            throw new Error("Can't find Livechat_guest_count setting");
        }
        return `guest-${livechatCount.value}`;
    }
    findByNameRegexWithExceptionsAndConditions(searchTerm, exceptions = [], conditions = {}, options = {}) {
        if (!Array.isArray(exceptions)) {
            exceptions = [exceptions];
        }
        const nameRegex = new RegExp(`^${(0, string_helpers_1.escapeRegExp)(searchTerm).trim()}`, 'i');
        const match = {
            $match: {
                name: nameRegex,
                _id: {
                    $nin: exceptions,
                },
                ...conditions,
            },
        };
        const { projection, sort, skip, limit } = options;
        const project = {
            $project: {
                // TODO: move this logic to client
                custom_name: { $concat: ['$username', ' - ', '$name'] },
                ...projection,
            },
        };
        const order = { $sort: sort || { name: 1 } };
        const params = [match, order, skip && { $skip: skip }, limit && { $limit: limit }, project].filter(Boolean);
        return this.col.aggregate(params);
    }
    /**
     * Find visitors by their email or phone or username or name
     */
    async findPaginatedVisitorsByEmailOrPhoneOrNameOrUsernameOrCustomField(emailOrPhone, nameOrUsername, allowedCustomFields = [], options) {
        if (!emailOrPhone && !nameOrUsername && allowedCustomFields.length === 0) {
            return this.findPaginated({ disabled: { $ne: true } }, options);
        }
        const query = {
            $or: [
                ...(emailOrPhone
                    ? [
                        {
                            'visitorEmails.address': emailOrPhone,
                        },
                        {
                            'phone.phoneNumber': emailOrPhone,
                        },
                    ]
                    : []),
                ...(nameOrUsername
                    ? [
                        {
                            name: nameOrUsername,
                        },
                        {
                            username: nameOrUsername,
                        },
                    ]
                    : []),
                ...allowedCustomFields.map((c) => ({ [`livechatData.${c}`]: nameOrUsername })),
            ],
            disabled: { $ne: true },
        };
        return this.findPaginated(query, options);
    }
    async findOneByEmailAndPhoneAndCustomField(email, phone, customFields) {
        const query = Object.assign({
            disabled: { $ne: true },
        }, {
            ...(email && { visitorEmails: { address: email } }),
            ...(phone && { phone: { phoneNumber: phone } }),
            ...customFields,
        });
        if (Object.keys(query).length === 1) {
            return null;
        }
        return this.findOne(query);
    }
    updateAllLivechatDataByToken(token, livechatDataToUpdate) {
        return this.updateOne({ token }, { $set: livechatDataToUpdate });
    }
    async updateLivechatDataByToken(token, key, value, overwrite = true) {
        const query = {
            token,
        };
        if (!overwrite) {
            const user = await this.getVisitorByToken(token, { projection: { livechatData: 1 } });
            if (user?.livechatData && typeof user.livechatData[key] !== 'undefined') {
                return true;
            }
        }
        const update = {
            $set: {
                [`livechatData.${key}`]: value,
            },
        }; // TODO: Remove this cast when TypeScript is updated
        // TypeScript is not smart enough to infer that `messages.${string}` matches keys of `ILivechatVisitor`;
        return this.updateOne(query, update);
    }
    updateLastAgentByToken(token, lastAgent) {
        const query = {
            token,
        };
        const update = {
            $set: {
                lastAgent,
            },
        };
        return this.updateOne(query, update);
    }
    updateById(_id, update) {
        return this.updateOne({ _id }, update);
    }
    async updateOneByIdOrToken(update, options) {
        let query = {};
        if (update._id) {
            query = { _id: update._id };
        }
        else if (update.token) {
            query = { token: update.token };
            update._id = new mongodb_1.ObjectId().toHexString();
        }
        return this.findOneAndUpdate(query, { $set: update }, options);
    }
    saveGuestById(_id, data) {
        const setData = {};
        const unsetData = {};
        if (data.name) {
            if (data.name?.trim()) {
                setData.name = data.name.trim();
            }
            else {
                unsetData.name = 1;
            }
        }
        if (data.email) {
            if (data.email?.trim()) {
                setData.visitorEmails = [{ address: data.email.trim() }];
            }
            else {
                unsetData.visitorEmails = 1;
            }
        }
        if (data.phone) {
            if (data.phone?.trim()) {
                setData.phone = [{ phoneNumber: data.phone.trim() }];
            }
            else {
                unsetData.phone = 1;
            }
        }
        if (data.livechatData) {
            Object.keys(data.livechatData).forEach((key) => {
                const value = data.livechatData[key]?.trim();
                if (value) {
                    setData[`livechatData.${key}`] = value;
                }
                else {
                    unsetData[`livechatData.${key}`] = 1;
                }
            });
        }
        const update = {
            ...(Object.keys(setData).length && { $set: setData }),
            ...(Object.keys(unsetData).length && { $unset: unsetData }),
        };
        if (!Object.keys(update).length) {
            return Promise.resolve(true);
        }
        return this.updateOne({ _id }, update);
    }
    removeDepartmentById(_id) {
        return this.updateOne({ _id }, { $unset: { department: 1 } });
    }
    removeById(_id) {
        return this.deleteOne({ _id });
    }
    saveGuestEmailPhoneById(_id, emails, phones) {
        const saveEmail = []
            .concat(emails)
            .filter((email) => email?.trim())
            .map((email) => ({ address: email }));
        const savePhone = []
            .concat(phones)
            .filter((phone) => phone?.trim().replace(/[^\d]/g, ''))
            .map((phone) => ({ phoneNumber: phone }));
        if (!saveEmail.length && !savePhone.length) {
            return Promise.resolve();
        }
        // the only reason we're using $setUnion here instead of $addToSet is because
        // old visitors might have `visitorEmails` or `phone` as `null` which would cause $addToSet to fail
        return this.updateOne({ _id }, [
            {
                $set: {
                    ...(saveEmail.length && { visitorEmails: { $setUnion: [{ $ifNull: ['$visitorEmails', []] }, saveEmail] } }),
                    ...(savePhone.length && { phone: { $setUnion: [{ $ifNull: ['$phone', []] }, savePhone] } }),
                },
            },
        ]);
    }
    removeContactManagerByUsername(manager) {
        return this.updateMany({
            contactManager: {
                username: manager,
            },
        }, {
            $unset: {
                contactManager: true,
            },
        });
    }
    disableById(_id) {
        return this.updateOne({ _id }, {
            $set: { disabled: true },
            $unset: {
                department: 1,
                contactManager: 1,
                token: 1,
                visitorEmails: 1,
                phone: 1,
                name: 1,
                livechatData: 1,
                lastChat: 1,
                ip: 1,
                host: 1,
                userAgent: 1,
                username: 1,
                ts: 1,
                status: 1,
            },
        });
    }
    setLastChatById(_id, lastChat) {
        return this.updateOne({ _id }, {
            $set: {
                lastChat,
            },
        });
    }
    updateDepartmentById(_id, department) {
        return this.findOneAndUpdate({ _id }, { $set: { department } }, { returnDocument: 'after' });
    }
}
exports.LivechatVisitorsRaw = LivechatVisitorsRaw;
//# sourceMappingURL=LivechatVisitors.js.map