"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VoipRoomRaw = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const logger_1 = require("@rocket.chat/logger");
const string_helpers_1 = require("@rocket.chat/string-helpers");
const BaseRaw_1 = require("./BaseRaw");
class VoipRoomRaw extends BaseRaw_1.BaseRaw {
    constructor(db, trash) {
        super(db, 'room', trash);
        this.logger = new logger_1.Logger('VoipRoomsRaw');
    }
    async findOneOpenByVisitorToken(visitorToken, options = {}) {
        const query = {
            't': 'v',
            'open': true,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    findOpenByAgentId(agentId) {
        return this.find({
            't': 'v',
            'open': true,
            'servedBy._id': agentId,
        });
    }
    async findOneByAgentId(agentId) {
        return this.findOne({
            't': 'v',
            'open': true,
            'servedBy._id': agentId,
        });
    }
    async findOneVoipRoomById(id, options = {}) {
        const query = {
            t: 'v',
            _id: id,
        };
        return this.findOne(query, options);
    }
    async findOneOpenByRoomIdAndVisitorToken(roomId, visitorToken, options = {}) {
        const query = {
            't': 'v',
            '_id': roomId,
            'open': true,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    async findOneByVisitorToken(visitorToken, options = {}) {
        const query = {
            't': 'v',
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    async findOneByIdAndVisitorToken(_id, visitorToken, options = {}) {
        const query = {
            't': 'v',
            _id,
            'v.token': visitorToken,
        };
        return this.findOne(query, options);
    }
    closeByRoomId(roomId, closeInfo) {
        const { closer, closedBy, closedAt, callDuration, serviceTimeDuration, ...extraData } = closeInfo;
        return this.updateOne({
            _id: roomId,
            t: 'v',
        }, {
            $set: {
                closer,
                closedBy,
                closedAt,
                callDuration,
                'metrics.serviceTimeDuration': serviceTimeDuration,
                'v.status': core_typings_1.UserStatus.OFFLINE,
                ...extraData,
            },
            $unset: {
                open: 1,
            },
        });
    }
    findRoomsWithCriteria({ agents, open, createdAt, closedAt, tags, queue, visitorId, direction, roomName, options = {}, }) {
        const query = {
            t: 'v',
            ...(visitorId && visitorId !== 'undefined' && { 'v._id': visitorId }),
            ...(agents && { 'servedBy._id': { $in: agents } }),
        };
        if (open !== undefined) {
            query.open = { $exists: open };
        }
        if (createdAt && Object.keys(createdAt).length) {
            query.ts = {};
            if (createdAt.start) {
                query.ts.$gte = new Date(createdAt.start);
            }
            if (createdAt.end) {
                query.ts.$lte = new Date(createdAt.end);
            }
        }
        if (closedAt && Object.keys(closedAt).length) {
            query.closedAt = {};
            if (closedAt.start) {
                query.closedAt.$gte = new Date(closedAt.start);
            }
            if (closedAt.end) {
                query.closedAt.$lte = new Date(closedAt.end);
            }
        }
        if (tags) {
            query.tags = { $in: tags };
        }
        if (queue) {
            query.queue = queue;
        }
        if (direction) {
            query.direction = direction;
        }
        if (roomName) {
            query.name = new RegExp((0, string_helpers_1.escapeRegExp)(roomName), 'i');
        }
        return this.findPaginated(query, {
            sort: options.sort || { name: 1 },
            skip: options.offset,
            limit: options.count,
        });
    }
}
exports.VoipRoomRaw = VoipRoomRaw;
//# sourceMappingURL=VoipRoom.js.map