import { compareBSONValues, getBSONType } from './bson';
import { isBinary, isIndexable, isPlainObject, isTruthy, equals } from './common';
import { createLookupFunction } from './lookups';
const everyMatches = (arr, fn) => arr.reduce((acc, item) => (acc.result ? fn(item) : acc), { result: true });
const someMatches = (arr, fn) => arr.reduce((acc, item) => (acc.result ? acc : fn(item)), { result: false });
const regexpElementMatcher = (regexp) => (value) => {
    if (value instanceof RegExp) {
        return value.toString() === regexp.toString();
    }
    if (typeof value !== 'string') {
        return false;
    }
    regexp.lastIndex = 0;
    return regexp.test(value);
};
const equalityElementMatcher = (elementSelector) => {
    if (isOperatorObject(elementSelector)) {
        throw new Error("Can't create equalityValueSelector for operator object");
    }
    if (elementSelector === null || elementSelector === undefined) {
        return (value) => value === null || value === undefined;
    }
    return (value) => equals(elementSelector, value);
};
const invertBranchedMatcher = (branchedMatcher) => (branches) => ({
    result: !branchedMatcher(branches).result,
});
const getValueBitmask = (value, length) => {
    if (typeof value === 'number' && Number.isSafeInteger(value)) {
        const buffer = new ArrayBuffer(Math.max(length, 2 * Uint32Array.BYTES_PER_ELEMENT));
        let view = new Uint32Array(buffer, 0, 2);
        view[0] = value % ((1 << 16) * (1 << 16)) | 0;
        view[1] = (value / ((1 << 16) * (1 << 16))) | 0;
        if (value < 0) {
            view = new Uint8Array(buffer, 2);
            view.forEach((_byte, i) => {
                view[i] = 0xff;
            });
        }
        return new Uint8Array(buffer);
    }
    if (isBinary(value)) {
        return new Uint8Array(value.buffer);
    }
    return false;
};
const getOperandBitmask = (operand, selector) => {
    if (typeof operand === 'number' && Number.isInteger(operand) && operand >= 0) {
        return new Uint8Array(new Int32Array([operand]).buffer);
    }
    if (isBinary(operand)) {
        return new Uint8Array(operand.buffer);
    }
    if (Array.isArray(operand) && operand.every((x) => Number.isInteger(x) && x >= 0)) {
        const buffer = new ArrayBuffer((Math.max(...operand) >> 3) + 1);
        const view = new Uint8Array(buffer);
        operand.forEach((x) => {
            view[x >> 3] |= 1 << (x & 0x7);
        });
        return view;
    }
    throw new Error(`operand to ${selector} must be a numeric bitmask (representable as a ` +
        'non-negative 32-bit signed integer), a bindata bitmask or an array with ' +
        'bit positions (non-negative integers)');
};
const expandArraysInBranches = (branches, skipTheArrays) => {
    const branchesOut = [];
    for (const branch of branches) {
        if (!(skipTheArrays && Array.isArray(branch.value) && !branch.dontIterate)) {
            branchesOut.push({ arrayIndices: branch.arrayIndices, value: branch.value });
        }
        if (Array.isArray(branch.value) && !branch.dontIterate) {
            branch.value.forEach((value, i) => {
                branchesOut.push({
                    arrayIndices: (branch.arrayIndices || []).concat(i),
                    value,
                });
            });
        }
    }
    return branchesOut;
};
const convertElementMatcherToBranchedMatcher = (elementMatcher, options = {}) => {
    return (branches) => {
        const expanded = options.dontExpandLeafArrays ? branches : expandArraysInBranches(branches, options.dontIncludeLeafArrays);
        return someMatches(expanded, (element) => {
            const matched = elementMatcher(element.value);
            if (typeof matched === 'number') {
                return {
                    result: true,
                    arrayIndices: [matched],
                };
            }
            return {
                result: matched,
                arrayIndices: element.arrayIndices,
            };
        });
    };
};
const operatorBranchedMatcher = (valueSelector) => {
    const operatorMatchers = Object.entries(valueSelector).map(([operator, operand]) => {
        if (isValueOperator(operator)) {
            return valueOperators[operator](operand, valueSelector);
        }
        if (isElementOperator(operator)) {
            const compileElementSelector = elementOperators[operator];
            return convertElementMatcherToBranchedMatcher(compileElementSelector(operand, valueSelector), {
                dontExpandLeafArrays: operator === '$size' || operator === '$elemMatch',
                dontIncludeLeafArrays: operator === '$type',
            });
        }
        throw new Error(`Unrecognized operator: ${operator}`);
    });
    return (branches) => everyMatches(operatorMatchers, (fn) => fn(branches));
};
const $in = (operand) => {
    if (!Array.isArray(operand)) {
        throw new Error('$in needs an array');
    }
    const elementMatchers = operand.map((option) => {
        if (option instanceof RegExp) {
            return regexpElementMatcher(option);
        }
        if (isOperatorObject(option)) {
            throw new Error('cannot nest $ under $in');
        }
        return equalityElementMatcher(option);
    });
    return (value) => {
        if (value === undefined) {
            value = null;
        }
        return elementMatchers.some((matcher) => matcher(value));
    };
};
const $eq = (operand) => convertElementMatcherToBranchedMatcher(equalityElementMatcher(operand));
const $not = (operand) => {
    return invertBranchedMatcher(compileValueSelector(operand));
};
const $ne = (operand) => invertBranchedMatcher(convertElementMatcherToBranchedMatcher(equalityElementMatcher(operand)));
const $nin = (operand) => invertBranchedMatcher(convertElementMatcherToBranchedMatcher($in(operand)));
const $exists = (operand) => {
    const exists = convertElementMatcherToBranchedMatcher((value) => value !== undefined);
    return operand ? exists : invertBranchedMatcher(exists);
};
const $options = (_operand, valueSelector) => {
    if (!('$regex' in valueSelector)) {
        throw new Error('$options needs a $regex');
    }
    return () => ({ result: true });
};
const $all = (operand) => {
    if (!Array.isArray(operand)) {
        throw new Error('$all requires array');
    }
    if (operand.length === 0) {
        return () => ({ result: false });
    }
    const branchedMatchers = operand.map((criterion) => {
        if (isOperatorObject(criterion)) {
            throw new Error('no $ expressions in $all');
        }
        return compileValueSelector(criterion);
    });
    return (branches) => everyMatches(branchedMatchers, (fn) => fn(branches));
};
const valueOperators = {
    $eq,
    $not,
    $ne,
    $nin,
    $exists,
    $options,
    $all,
};
const isValueOperator = (operator) => operator in valueOperators;
function createInequalityOperator(selector) {
    return (operand) => {
        if (Array.isArray(operand)) {
            return () => false;
        }
        operand !== null && operand !== void 0 ? operand : (operand = null);
        const operandType = getBSONType(operand);
        return (value) => {
            value !== null && value !== void 0 ? value : (value = null);
            if (getBSONType(value) !== operandType) {
                return false;
            }
            return selector(compareBSONValues(value, operand));
        };
    };
}
const $lt = createInequalityOperator((cmpValue) => cmpValue < 0);
const $gt = createInequalityOperator((cmpValue) => cmpValue > 0);
const $lte = createInequalityOperator((cmpValue) => cmpValue <= 0);
const $gte = createInequalityOperator((cmpValue) => cmpValue >= 0);
const $mod = (operand) => {
    if (!(Array.isArray(operand) && operand.length === 2 && typeof operand[0] === 'number' && typeof operand[1] === 'number')) {
        throw new Error('argument to $mod must be an array of two numbers');
    }
    const [divisor, remainder] = operand;
    return (value) => typeof value === 'number' && value % divisor === remainder;
};
const $size = (operand) => {
    if (typeof operand === 'string') {
        operand = 0;
    }
    else if (typeof operand !== 'number') {
        throw new Error('$size needs a number');
    }
    return (value) => Array.isArray(value) && value.length === operand;
};
const $type = (operand) => {
    if (typeof operand === 'string') {
        const operandAliasMap = {
            double: 1,
            string: 2,
            object: 3,
            array: 4,
            binData: 5,
            undefined: 6,
            objectId: 7,
            bool: 8,
            date: 9,
            null: 10,
            regex: 11,
            dbPointer: 12,
            javascript: 13,
            symbol: 14,
            javascriptWithScope: 15,
            int: 16,
            timestamp: 17,
            long: 18,
            decimal: 19,
            minKey: -1,
            maxKey: 127,
        };
        if (!(operand in operandAliasMap)) {
            throw new Error(`unknown string alias for $type: ${operand}`);
        }
        operand = operandAliasMap[operand];
    }
    else if (typeof operand === 'number') {
        if (operand === 0 || operand < -1 || (operand > 19 && operand !== 127)) {
            throw new Error(`Invalid numerical $type code: ${operand}`);
        }
    }
    else {
        throw new Error('argument to $type is not a number or a string');
    }
    return (value) => value !== undefined && getBSONType(value) === operand;
};
const $bitsAllSet = (operand) => {
    const mask = getOperandBitmask(operand, '$bitsAllSet');
    return (value) => {
        const bitmask = getValueBitmask(value, mask.length);
        return bitmask && mask.every((byte, i) => (bitmask[i] & byte) === byte);
    };
};
const $bitsAnySet = (operand) => {
    const mask = getOperandBitmask(operand, '$bitsAnySet');
    return (value) => {
        const bitmask = getValueBitmask(value, mask.length);
        return bitmask && mask.some((byte, i) => (~bitmask[i] & byte) !== byte);
    };
};
const $bitsAllClear = (operand) => {
    const mask = getOperandBitmask(operand, '$bitsAllClear');
    return (value) => {
        const bitmask = getValueBitmask(value, mask.length);
        return bitmask && mask.every((byte, i) => !(bitmask[i] & byte));
    };
};
const $bitsAnyClear = (operand) => {
    const mask = getOperandBitmask(operand, '$bitsAnyClear');
    return (value) => {
        const bitmask = getValueBitmask(value, mask.length);
        return bitmask && mask.some((byte, i) => (bitmask[i] & byte) !== byte);
    };
};
const $regex = (operand, valueSelector) => {
    if (!(typeof operand === 'string' || operand instanceof RegExp)) {
        throw new Error('$regex has to be a string or RegExp');
    }
    if (valueSelector.$options !== undefined) {
        if (/[^gim]/.test(valueSelector.$options)) {
            throw new Error('Only the i, m, and g regexp options are supported');
        }
        const source = operand instanceof RegExp ? operand.source : operand;
        return regexpElementMatcher(new RegExp(source, valueSelector.$options));
    }
    if (operand instanceof RegExp) {
        return regexpElementMatcher(operand);
    }
    return regexpElementMatcher(new RegExp(operand));
};
const $elemMatch = (operand) => {
    if (!isPlainObject(operand)) {
        throw new Error('$elemMatch need an object');
    }
    const isDocMatcher = !isOperatorObject(Object.entries(operand)
        .filter(([key]) => !isLogicalOperator(key))
        .reduce((a, [key, value]) => Object.assign(a, { [key]: value }), {}), true);
    if (isDocMatcher) {
        const subMatcher = createDocumentMatcherFromFilter(operand);
        return (value) => {
            if (!Array.isArray(value)) {
                return false;
            }
            for (let i = 0; i < value.length; ++i) {
                const arrayElement = value[i];
                if (!isIndexable(arrayElement)) {
                    return false;
                }
                if (subMatcher(arrayElement).result) {
                    return i;
                }
            }
            return false;
        };
    }
    const subMatcher = compileValueSelector(operand);
    return (value) => {
        if (!Array.isArray(value)) {
            return false;
        }
        for (let i = 0; i < value.length; ++i) {
            const arrayElement = value[i];
            if (subMatcher([{ value: arrayElement, dontIterate: true }]).result) {
                return i;
            }
        }
        return false;
    };
};
const elementOperators = {
    $lt,
    $gt,
    $lte,
    $gte,
    $mod,
    $in,
    $size,
    $type,
    $bitsAllSet,
    $bitsAnySet,
    $bitsAllClear,
    $bitsAnyClear,
    $regex,
    $elemMatch,
};
const isElementOperator = (operator) => operator in elementOperators;
const $and = (subSelector) => {
    const matchers = compileArrayOfDocumentSelectors(subSelector);
    return (doc) => everyMatches(matchers, (fn) => fn(doc));
};
const $or = (subSelector) => {
    const matchers = compileArrayOfDocumentSelectors(subSelector);
    return (doc) => ({ result: matchers.some((fn) => fn(doc).result) });
};
const $nor = (subSelector) => {
    const matchers = compileArrayOfDocumentSelectors(subSelector);
    return (doc) => ({ result: matchers.every((fn) => !fn(doc).result) });
};
const $where = (selectorValue) => {
    if (!(selectorValue instanceof Function)) {
        selectorValue = Function('obj', `return ${selectorValue}`);
    }
    return (doc) => ({ result: selectorValue.call(doc, doc) });
};
const logicalOperators = {
    $and,
    $or,
    $nor,
    $where,
};
const isLogicalOperator = (operator) => operator in logicalOperators;
const isOperatorObject = (valueSelector, inconsistentOK = false) => {
    if (!isPlainObject(valueSelector)) {
        return false;
    }
    let theseAreOperators = undefined;
    for (const selKey of Object.keys(valueSelector)) {
        const thisIsOperator = selKey.slice(0, 1) === '$' || selKey === 'diff';
        if (theseAreOperators === undefined) {
            theseAreOperators = thisIsOperator;
        }
        else if (theseAreOperators !== thisIsOperator) {
            if (!inconsistentOK) {
                throw new Error(`Inconsistent operator: ${JSON.stringify(valueSelector)}`);
            }
            theseAreOperators = false;
        }
    }
    return theseAreOperators !== null && theseAreOperators !== void 0 ? theseAreOperators : true;
};
const compileValueSelector = (valueSelector) => {
    if (valueSelector instanceof RegExp) {
        return convertElementMatcherToBranchedMatcher(regexpElementMatcher(valueSelector));
    }
    if (isOperatorObject(valueSelector)) {
        return operatorBranchedMatcher(valueSelector);
    }
    return convertElementMatcherToBranchedMatcher(equalityElementMatcher(valueSelector));
};
const compileArrayOfDocumentSelectors = (selectors) => {
    return selectors.map((subSelector) => createDocumentMatcherFromFilter(subSelector));
};
export const createDocumentMatcherFromFilter = (filter) => {
    const docMatchers = Object.entries(filter)
        .map(([key, subSelector]) => {
        if (isLogicalOperator(key)) {
            return logicalOperators[key](subSelector);
        }
        if (key.slice(0, 1) === '$') {
            throw new Error(`Unrecognized logical operator: ${key}`);
        }
        if (typeof subSelector === 'function') {
            return undefined;
        }
        const lookUpByIndex = createLookupFunction(key);
        const valueMatcher = compileValueSelector(subSelector);
        return (doc) => valueMatcher(lookUpByIndex(doc));
    })
        .filter(isTruthy);
    return (doc) => everyMatches(docMatchers, (fn) => fn(doc));
};
export const createPredicateFromFilter = (filter) => {
    const docMatcher = createDocumentMatcherFromFilter(filter);
    return (doc) => docMatcher(doc).result;
};
//# sourceMappingURL=filter.js.map