"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerGuest = void 0;
const core_typings_1 = require("@rocket.chat/core-typings");
const logger_1 = require("@rocket.chat/logger");
const models_1 = require("@rocket.chat/models");
const patch_injection_1 = require("@rocket.chat/patch-injection");
const tools_1 = require("@rocket.chat/tools");
const logger = new logger_1.Logger('Livechat - Visitor');
exports.registerGuest = (0, patch_injection_1.makeFunction)(async ({ id, token, name, phone, email, department, username, connectionData, status = core_typings_1.UserStatus.ONLINE }, { shouldConsiderIdleAgent }) => {
    if (!token) {
        throw Error('error-invalid-token');
    }
    logger.debug(`New incoming conversation: id: ${id} | token: ${token}`);
    const visitorDataToUpdate = {
        token,
        status,
        ...(phone?.number && { phone: [{ phoneNumber: phone.number }] }),
        ...(name && { name }),
    };
    if (email) {
        const visitorEmail = email.trim().toLowerCase();
        (0, tools_1.validateEmail)(visitorEmail);
        visitorDataToUpdate.visitorEmails = [{ address: visitorEmail }];
        const contact = await models_1.LivechatContacts.findContactByEmailAndContactManager(visitorEmail);
        if (contact?.contactManager) {
            const agent = await models_1.Users.findOneOnlineAgentById(contact.contactManager, shouldConsiderIdleAgent, {
                projection: { _id: 1, username: 1, name: 1, emails: 1 },
            });
            if (agent?.username && agent.name && agent.emails) {
                visitorDataToUpdate.contactManager = {
                    _id: agent._id,
                    username: agent.username,
                    name: agent.name,
                    emails: agent.emails,
                };
                logger.debug(`Assigning visitor ${token} to agent ${agent.username}`);
            }
        }
    }
    const livechatVisitor = await models_1.LivechatVisitors.getVisitorByToken(token, { projection: { _id: 1 } });
    if (department && livechatVisitor?.department !== department) {
        logger.debug(`Attempt to find a department with id/name ${department}`);
        const dep = await models_1.LivechatDepartment.findOneByIdOrName(department, { projection: { _id: 1 } });
        if (!dep) {
            logger.debug(`Invalid department provided: ${department}`);
            // throw new Meteor.Error('error-invalid-department', 'The provided department is invalid');
            throw new Error('error-invalid-department');
        }
        logger.debug(`Assigning visitor ${token} to department ${dep._id}`);
        visitorDataToUpdate.department = dep._id;
    }
    visitorDataToUpdate.token = livechatVisitor?.token || token;
    let existingUser = null;
    if (livechatVisitor) {
        logger.debug('Found matching user by token');
        visitorDataToUpdate._id = livechatVisitor._id;
    }
    else if (phone?.number && (existingUser = await models_1.LivechatVisitors.findOneVisitorByPhone(phone.number))) {
        logger.debug('Found matching user by phone number');
        visitorDataToUpdate._id = existingUser._id;
        // Don't change token when matching by phone number, use current visitor token
        visitorDataToUpdate.token = existingUser.token;
    }
    else if (email && (existingUser = await models_1.LivechatVisitors.findOneGuestByEmailAddress(email))) {
        logger.debug('Found matching user by email');
        visitorDataToUpdate._id = existingUser._id;
    }
    else if (!livechatVisitor) {
        logger.debug(`No matches found. Attempting to create new user with token ${token}`);
        visitorDataToUpdate._id = id || undefined;
        visitorDataToUpdate.username = username || (await models_1.LivechatVisitors.getNextVisitorUsername());
        visitorDataToUpdate.status = status;
        visitorDataToUpdate.ts = new Date();
        if (connectionData && typeof connectionData === 'object') {
            logger.debug(`Saving connection data for visitor ${token}`);
            const { httpHeaders, clientAddress } = connectionData;
            if (httpHeaders && typeof httpHeaders === 'object') {
                visitorDataToUpdate.userAgent = httpHeaders['user-agent'];
                visitorDataToUpdate.ip = httpHeaders['x-real-ip'] || httpHeaders['x-forwarded-for'] || clientAddress;
                visitorDataToUpdate.host = httpHeaders.host;
            }
        }
    }
    const upsertedLivechatVisitor = await models_1.LivechatVisitors.updateOneByIdOrToken(visitorDataToUpdate, {
        upsert: true,
        returnDocument: 'after',
    });
    if (!upsertedLivechatVisitor) {
        logger.debug(`No visitor found after upsert`);
        return null;
    }
    return upsertedLivechatVisitor;
});
//# sourceMappingURL=create.js.map