"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Presence = void 0;
const core_services_1 = require("@rocket.chat/core-services");
const core_typings_1 = require("@rocket.chat/core-typings");
const models_1 = require("@rocket.chat/models");
const PresenceReaper_1 = require("./lib/PresenceReaper");
const processConnectionStatus_1 = require("./lib/processConnectionStatus");
const MAX_CONNECTIONS = 200;
class Presence extends core_services_1.ServiceClass {
    constructor() {
        super();
        this.name = 'presence';
        this.broadcastEnabled = true;
        this.hasPresenceLicense = false;
        this.hasScalabilityLicense = false;
        this.hasLicense = false;
        this.connsPerInstance = new Map();
        this.peakConnections = 0;
        this.reaper = new PresenceReaper_1.PresenceReaper({
            batchSize: 500,
            staleThresholdMs: 5 * 60 * 1000, // 5 minutes
            onUpdate: (userIds) => this.handleReaperUpdates(userIds),
        });
        this.onEvent('watch.instanceStatus', async ({ clientAction, id, diff }) => {
            if (clientAction === 'removed') {
                this.connsPerInstance.delete(id);
                const affectedUsers = await this.removeLostConnections(id);
                affectedUsers.forEach((uid) => this.updateUserPresence(uid));
                return;
            }
            // always store the number of connections per instance so we can show correct in the UI
            if (diff?.hasOwnProperty('extraInformation.conns')) {
                this.connsPerInstance.set(id, diff['extraInformation.conns']);
                this.peakConnections = Math.max(this.peakConnections, this.getTotalConnections());
                this.validateAvailability();
            }
        });
        this.onEvent('license.module', async ({ module, valid }) => {
            switch (module) {
                case 'unlimited-presence':
                    this.hasPresenceLicense = valid;
                    break;
                case 'scalability':
                    this.hasScalabilityLicense = valid;
                    break;
                default:
                    return;
            }
            // The scalability module is also accepted as a way to enable the presence service for backwards compatibility
            this.hasLicense = this.hasPresenceLicense || this.hasScalabilityLicense;
            // broadcast should always be enabled if license is active (unless the troubleshoot setting is on)
            if (!this.broadcastEnabled && this.hasLicense) {
                await this.toggleBroadcast(true);
            }
        });
    }
    async onNodeDisconnected({ node }) {
        const affectedUsers = await this.removeLostConnections(node.id);
        return affectedUsers.forEach((uid) => this.updateUserPresence(uid));
    }
    async started() {
        this.reaper.start();
        this.lostConTimeout = setTimeout(async () => {
            const affectedUsers = await this.removeLostConnections();
            return affectedUsers.forEach((uid) => this.updateUserPresence(uid));
        }, 10000);
        try {
            await models_1.Settings.updateValueById('Presence_broadcast_disabled', false);
            this.hasScalabilityLicense = await core_services_1.License.hasModule('scalability');
            this.hasPresenceLicense = await core_services_1.License.hasModule('unlimited-presence');
            this.hasLicense = this.hasPresenceLicense || this.hasScalabilityLicense;
        }
        catch (e) {
            // ignore
        }
    }
    async handleReaperUpdates(userIds) {
        const results = await Promise.allSettled(userIds.map((uid) => this.updateUserPresence(uid)));
        const fulfilled = results.filter((result) => result.status === 'fulfilled');
        const rejected = results.filter((result) => result.status === 'rejected');
        if (fulfilled.length > 0) {
            console.debug(`[PresenceReaper] Successfully updated presence for ${fulfilled.length} users.`);
        }
        if (rejected.length > 0) {
            console.error(`[PresenceReaper] Failed to update presence for ${rejected.length} users:`, rejected.map(({ reason }) => reason));
        }
    }
    async stopped() {
        this.reaper.stop();
        if (!this.lostConTimeout) {
            return;
        }
        clearTimeout(this.lostConTimeout);
    }
    async toggleBroadcast(enabled) {
        if (!this.hasLicense && this.getTotalConnections() > MAX_CONNECTIONS) {
            throw new Error('Cannot enable broadcast when there are more than 200 connections');
        }
        this.broadcastEnabled = enabled;
        // update the setting only to turn it on, because it may have been disabled via the troubleshooting setting, which doesn't affect the setting
        if (enabled) {
            await models_1.Settings.updateValueById('Presence_broadcast_disabled', false);
        }
    }
    getConnectionCount() {
        return {
            current: this.getTotalConnections(),
            max: MAX_CONNECTIONS,
        };
    }
    async newConnection(uid, session, nodeId) {
        if (!uid || !session) {
            return;
        }
        await models_1.UsersSessions.addConnectionById(uid, {
            id: session,
            instanceId: nodeId,
            status: core_typings_1.UserStatus.ONLINE,
        });
        await this.updateUserPresence(uid);
        return {
            uid,
            connectionId: session,
        };
    }
    async updateConnection(uid, connectionId) {
        const query = {
            '_id': uid,
            'connections.id': connectionId,
        };
        const update = {
            $set: {
                'connections.$._updatedAt': new Date(),
            },
        };
        const result = await models_1.UsersSessions.updateOne(query, update);
        if (result.modifiedCount === 0) {
            return;
        }
        await this.updateUserPresence(uid);
        return { uid, connectionId };
    }
    async removeConnection(uid, session) {
        if (!uid || !session) {
            return;
        }
        await models_1.UsersSessions.removeConnectionByConnectionId(session);
        await this.updateUserPresence(uid);
        return {
            uid,
            session,
        };
    }
    async removeLostConnections(nodeID) {
        if (nodeID) {
            const affectedUsers = await models_1.UsersSessions.findByInstanceId(nodeID).toArray();
            const { modifiedCount } = await models_1.UsersSessions.removeConnectionsFromInstanceId(nodeID);
            if (modifiedCount === 0) {
                return [];
            }
            return affectedUsers.map(({ _id }) => _id);
        }
        const nodes = (await this.api?.nodeList()) || [];
        const ids = nodes.filter((node) => node.available).map(({ id }) => id);
        if (ids.length === 0) {
            return [];
        }
        const affectedUsers = await models_1.UsersSessions.findByOtherInstanceIds(ids, { projection: { _id: 1 } }).toArray();
        const { modifiedCount } = await models_1.UsersSessions.removeConnectionsFromOtherInstanceIds(ids);
        if (modifiedCount === 0) {
            return [];
        }
        return affectedUsers.map(({ _id }) => _id);
    }
    async setStatus(uid, statusDefault, statusText) {
        const userSessions = (await models_1.UsersSessions.findOneById(uid)) || { connections: [] };
        const user = await models_1.Users.findOneById(uid, {
            projection: { username: 1, roles: 1, status: 1 },
        });
        const { status, statusConnection } = (0, processConnectionStatus_1.processPresenceAndStatus)(userSessions.connections, statusDefault);
        const result = await models_1.Users.updateStatusById(uid, {
            statusDefault,
            status,
            statusConnection,
            statusText,
        });
        if (result.modifiedCount > 0) {
            this.broadcast({ _id: uid, username: user?.username, status, statusText, roles: user?.roles || [] }, user?.status);
        }
        return !!result.modifiedCount;
    }
    async setConnectionStatus(uid, status, session) {
        const result = await models_1.UsersSessions.updateConnectionStatusById(uid, session, status);
        await this.updateUserPresence(uid);
        return !!result.modifiedCount;
    }
    async updateUserPresence(uid) {
        const user = await models_1.Users.findOneById(uid, {
            projection: {
                username: 1,
                statusDefault: 1,
                statusText: 1,
                roles: 1,
                status: 1,
            },
        });
        if (!user) {
            return;
        }
        const userSessions = (await models_1.UsersSessions.findOneById(uid)) || { connections: [] };
        const { statusDefault } = user;
        const { status, statusConnection } = (0, processConnectionStatus_1.processPresenceAndStatus)(userSessions.connections, statusDefault);
        const result = await models_1.Users.updateStatusById(uid, {
            status,
            statusConnection,
        });
        if (result.modifiedCount > 0) {
            this.broadcast({ _id: uid, username: user.username, status, statusText: user.statusText, roles: user.roles }, user.status);
        }
    }
    broadcast(user, previousStatus) {
        if (!this.broadcastEnabled) {
            return;
        }
        this.api?.broadcast('presence.status', {
            user,
            previousStatus,
        });
    }
    async validateAvailability() {
        if (this.hasLicense) {
            return;
        }
        if (this.getTotalConnections() > MAX_CONNECTIONS) {
            this.broadcastEnabled = false;
            await models_1.Settings.updateValueById('Presence_broadcast_disabled', true);
        }
    }
    getTotalConnections() {
        return Array.from(this.connsPerInstance.values()).reduce((acc, conns) => acc + conns, 0);
    }
    getPeakConnections(reset = false) {
        const peak = this.peakConnections;
        if (reset) {
            this.resetPeakConnections();
        }
        return peak;
    }
    resetPeakConnections() {
        this.peakConnections = 0;
    }
}
exports.Presence = Presence;
//# sourceMappingURL=Presence.js.map