import type { IRoom, ISubscription, IUser } from '@rocket.chat/core-typings';
import type { OffCallbackHandler } from '@rocket.chat/emitter';
import type { ObjectId, Filter, FindOptions as MongoFindOptions, Document } from 'mongodb';
import { createContext } from 'react';

import type { SubscriptionWithRoom } from './types/SubscriptionWithRoom';

export type SubscriptionQuery =
	| {
			rid: string | ObjectId;
	  }
	| {
			name: string;
	  }
	| {
			open: boolean;
	  }
	| object;

export type Fields<TSchema extends Document = Document> = Exclude<MongoFindOptions<TSchema>['projection'], undefined>;

export type Sort<TSchema extends Document = Document> = Exclude<MongoFindOptions<TSchema>['sort'], undefined>;

export type FindOptions<TSchema extends Document = Document> = {
	fields?: Fields<TSchema>;
	sort?: Sort<TSchema>;
	skip?: number;
	limit?: number;
};

export type UserContextValue = {
	userId: string | undefined;
	user: IUser | null;
	queryPreference: <T>(
		key: string | ObjectId,
		defaultValue?: T,
	) => [subscribe: (onStoreChange: () => void) => () => void, getSnapshot: () => T | undefined];
	querySubscription: (
		query: Filter<Pick<ISubscription, 'rid' | 'name'>>,
	) => [subscribe: (onStoreChange: () => void) => () => void, getSnapshot: () => ISubscription | undefined];
	queryRoom: (
		query: Filter<Pick<IRoom, '_id'>>,
		fields?: Fields,
		sort?: Sort,
	) => [subscribe: (onStoreChange: () => void) => () => void, getSnapshot: () => IRoom | undefined];
	querySubscriptions: (
		query: SubscriptionQuery,
		options?: FindOptions,
	) => [subscribe: (onStoreChange: () => void) => () => void, getSnapshot: () => SubscriptionWithRoom[]];
	logout: () => Promise<void>;
	onLogout: (callback: () => void) => OffCallbackHandler;
};

export const UserContext = createContext<UserContextValue>({
	userId: undefined,
	user: null,
	queryPreference: () => [() => (): void => undefined, (): undefined => undefined],
	querySubscription: () => [() => (): void => undefined, (): undefined => undefined],
	queryRoom: () => [() => (): void => undefined, (): undefined => undefined],
	querySubscriptions: () => [() => (): void => undefined, (): [] => []],
	logout: () => Promise.resolve(),
	onLogout: () => (): void => undefined,
});
