import { Emitter } from '@rocket.chat/emitter';
import { useSafeRefCallback } from '@rocket.chat/ui-client';
import { useCallback, useRef, useState } from 'react';
const GRAB_DOM_EVENTS = ['pointerdown'];
const RELEASE_DOM_EVENTS = ['pointerup', 'pointercancel', 'lostpointercapture'];
const MOVE_DOM_EVENTS = ['pointermove'];
class GenericRect {
    constructor(rect) {
        this.rect = rect;
    }
    get x() {
        return this.rect.x;
    }
    get y() {
        return this.rect.y;
    }
    get width() {
        return this.rect.width;
    }
    get height() {
        return this.rect.height;
    }
    get top() {
        return this.rect.y;
    }
    get left() {
        return this.rect.x;
    }
    get right() {
        return this.rect.x + this.rect.width;
    }
    get bottom() {
        return this.rect.y + this.rect.height;
    }
}
class Draggable extends Emitter {
    constructor(element) {
        super();
        this.isDragging = false;
        this.pointerCoordinates = { x: 0, y: 0 };
        this.storedPositionOffset = { x: 0, y: 0 };
        this._element = element;
        this._element.onMove((pointerCoordinates) => {
            this.handleMove(pointerCoordinates);
        });
        this._element.onRelease((rect) => {
            this.handleRelease(rect);
        });
        this._element.onChangeView((draggableElement) => {
            this.emit('changeView', draggableElement);
        });
        this._element.onResize((rect) => {
            this.emit('resize', rect);
        });
    }
    get element() {
        return this._element;
    }
    onRelease(cb) {
        return this.on('release', cb);
    }
    onMove(cb) {
        return this.on('move', cb);
    }
    onResize(cb) {
        return this.on('resize', cb);
    }
    onChangeView(cb) {
        return this.on('changeView', cb);
    }
    setPointerCoordinates(pointerCoordinates) {
        this.pointerCoordinates = pointerCoordinates;
    }
    addElementPositionOffset(x, y) {
        const currentOffset = this.getStoredOffset();
        this.setStoredPositionOffset(currentOffset.x + x, currentOffset.y + y);
    }
    setStoredPositionOffset(x, y) {
        this.storedPositionOffset = { x, y };
        this.element.setElementPositionOffset(this.storedPositionOffset);
    }
    moveToCoordinates(targetElementCoordinates, initialPosition) {
        this.setStoredPositionOffset(targetElementCoordinates.x - initialPosition.x, targetElementCoordinates.y - initialPosition.y);
        this.emit('move', this.getStoredOffset());
    }
    handleGrab(startingPointerCoordinates, elementRect) {
        this.isDragging = true;
        this.setPointerCoordinates(startingPointerCoordinates);
        this.emit('grab', elementRect);
    }
    handleMove(currentPointerCoordinates) {
        if (!this.isDragging)
            return;
        const xDelta = currentPointerCoordinates.x - this.pointerCoordinates.x;
        const yDelta = currentPointerCoordinates.y - this.pointerCoordinates.y;
        this.addElementPositionOffset(xDelta, yDelta);
        this.setPointerCoordinates(currentPointerCoordinates);
        const storedOffset = this.getStoredOffset();
        this.element.setElementPositionOffset(storedOffset);
        this.emit('move', storedOffset);
    }
    handleRelease(finalElementPosition) {
        if (!this.isDragging) {
            return;
        }
        this.isDragging = false;
        this.setPointerCoordinates({ x: 0, y: 0 });
        this.emit('release', finalElementPosition);
    }
    moveByOffset(offset) {
        this.addElementPositionOffset(offset.x, offset.y);
        this.emit('move', this.getStoredOffset());
    }
    getStoredOffset() {
        return this.storedPositionOffset;
    }
}
export const DEFAULT_BOUNDING_ELEMENT_OPTIONS = {
    resizeDebounce: 150,
};
class BoundingElement extends Emitter {
    constructor(element, draggableInstance, options = DEFAULT_BOUNDING_ELEMENT_OPTIONS) {
        super();
        this.timeout = null;
        this._element = element;
        this.draggableInstance = draggableInstance;
        this.resizeDebounce = options.resizeDebounce;
        this.draggableInstance.onRelease(() => {
            this.tryMoveToBounds();
        });
        this.element.onResize(() => {
            this.tryMoveToBounds();
        });
        this.draggableInstance.onResize(() => {
            this.tryMoveToBounds();
        });
        this.draggableInstance.onChangeView(() => {
            this.tryMoveToBounds();
        });
        this.element.onChangeView(() => {
            this.tryMoveToBounds();
        });
    }
    get element() {
        return this._element;
    }
    _tryMoveToBounds() {
        const draggableRect = this.draggableInstance.element.getElementRect();
        const boundsRect = this.element.getElementRect();
        if (!draggableRect || !boundsRect) {
            return;
        }
        const offset = this.calculateBoundsOffset(draggableRect, boundsRect);
        this.draggableInstance.moveByOffset(offset);
    }
    tryMoveToBounds() {
        if (this.timeout) {
            clearTimeout(this.timeout);
        }
        this.timeout = setTimeout(() => {
            this._tryMoveToBounds();
        }, this.resizeDebounce);
    }
    calculateBoundsOffset(_draggableRect, _boundsRect) {
        const draggableRect = new GenericRect(_draggableRect);
        const boundsRect = new GenericRect(_boundsRect);
        // If the draggable element's top/left position is less than
        // the bounding element's top/left position, the difference will be positive
        // This means we can just add this value to the draggable element's offset
        // to bring it back into bounds
        const topLeftPoint = {
            x: boundsRect.left - draggableRect.left,
            y: boundsRect.top - draggableRect.top,
        };
        // If the draggable element's bottom/right position is greater than
        // the bounding element's bottom/right position, the difference will be negative
        // This means we can just add this value to the draggable element's offset
        // to bring it back into bounds
        const bottomRightPoint = {
            x: boundsRect.right - draggableRect.right,
            y: boundsRect.bottom - draggableRect.bottom,
        };
        const offset = {
            x: 0,
            y: 0,
        };
        if (topLeftPoint.x > 0) {
            offset.x += topLeftPoint.x;
        }
        if (topLeftPoint.y > 0) {
            offset.y += topLeftPoint.y;
        }
        if (bottomRightPoint.x < 0) {
            offset.x += bottomRightPoint.x;
        }
        if (bottomRightPoint.y < 0) {
            offset.y += bottomRightPoint.y;
        }
        return offset;
    }
}
class HandleElement extends Emitter {
    constructor(element, draggableInstance) {
        super();
        this.draggableInstance = draggableInstance;
        this._element = element;
        this._element.onGrab(([mousePosition, elementRect]) => {
            this.draggableInstance.handleGrab(mousePosition, elementRect);
        });
    }
    get element() {
        return this._element;
    }
}
const getPointerEventCoordinates = (e) => ({
    x: e.clientX,
    y: e.clientY,
});
const isLeftClick = (event) => event.button === 0;
const isMousePointer = (event) => event.pointerType === 'mouse';
class HandleDomElement extends Emitter {
    constructor() {
        super(...arguments);
        this.onGrab = (cb) => {
            return this.on('grab', cb);
        };
    }
    setElement(element) {
        const onGrab = (event) => {
            const element = event.currentTarget;
            if (!element || (isMousePointer(event) && !isLeftClick(event))) {
                return;
            }
            event.preventDefault();
            this.emit('grab', [getPointerEventCoordinates(event), element.getBoundingClientRect()]);
        };
        const unsubArray = [];
        unsubArray.push(...GRAB_DOM_EVENTS.map((event) => {
            element.addEventListener(event, onGrab);
            return () => element.removeEventListener(event, onGrab);
        }));
        return () => unsubArray.forEach((unsub) => unsub());
    }
}
class BoundingDomElement extends Emitter {
    constructor() {
        super(...arguments);
        this._element = null;
        this.onResize = (cb) => {
            return this.on('resize', cb);
        };
        this.onChangeView = (cb) => {
            return this.on('changeView', cb);
        };
    }
    setElement(element) {
        this._element = element;
        const onResize = (entries) => {
            const firstEntry = entries[0];
            if (!firstEntry) {
                return;
            }
            this.emit('resize', firstEntry.contentRect);
        };
        const observer = new ResizeObserver(onResize);
        observer.observe(element);
        this.emit('changeView', this);
        return () => {
            observer.disconnect();
            this._element = null;
        };
    }
    getElementRect() {
        if (!this._element) {
            return null;
        }
        return this._element.getBoundingClientRect();
    }
}
class DraggableDomElement extends Emitter {
    constructor() {
        super(...arguments);
        this.element = null;
        this.onResize = (cb) => {
            return this.on('resize', cb);
        };
        this.onChangeView = (cb) => {
            return this.on('changeView', cb);
        };
        this.onMove = (cb) => {
            return this.on('move', cb);
        };
        this.onRelease = (cb) => {
            return this.on('release', cb);
        };
    }
    setElement(element) {
        this.element = element;
        const onEnd = () => {
            const elementRect = this.getElementRect();
            if (!elementRect) {
                return;
            }
            this.emit('release', elementRect);
        };
        const onMove = (event) => {
            this.emit('move', getPointerEventCoordinates(event));
        };
        const onResize = (entries) => {
            const firstEntry = entries[0];
            if (!firstEntry) {
                return;
            }
            this.emit('resize', firstEntry.contentRect);
        };
        const unsubArray = [];
        // Attach MOVE DOM listeners
        unsubArray.push(...MOVE_DOM_EVENTS.map((event) => {
            window.addEventListener(event, onMove);
            return () => window.removeEventListener(event, onMove);
        }));
        // Attach RELEASE DOM listeners
        unsubArray.push(...RELEASE_DOM_EVENTS.map((event) => {
            window.addEventListener(event, onEnd);
            return () => window.removeEventListener(event, onEnd);
        }));
        const observer = new ResizeObserver(onResize);
        observer.observe(element);
        unsubArray.push(() => observer.disconnect());
        this.emit('changeView', this);
        return () => {
            this.element = null;
            unsubArray.forEach((unsub) => unsub());
        };
    }
    getElementRect() {
        if (!this.element) {
            return null;
        }
        return this.element.getBoundingClientRect();
    }
    setElementPositionOffset(offset) {
        if (!this.element) {
            return;
        }
        this.element.style.transform = `translate(${offset.x}px, ${offset.y}px)`;
    }
}
export const useDraggable = () => {
    const [draggableElement] = useState(() => new Draggable(new DraggableDomElement()));
    const [boundingElement] = useState(() => new BoundingElement(new BoundingDomElement(), draggableElement));
    const [handleElement] = useState(() => new HandleElement(new HandleDomElement(), draggableElement));
    const restorePositionRef = useRef(null);
    const handleElementCallbackRef = useSafeRefCallback(useCallback((node) => {
        if (!node) {
            return;
        }
        return handleElement.element.setElement(node);
    }, [handleElement]));
    const draggableCallbackRef = useSafeRefCallback(useCallback((node) => {
        if (!node) {
            return;
        }
        const offMove = draggableElement.onMove(() => {
            restorePositionRef.current = node.getBoundingClientRect();
        });
        const offDomEvents = draggableElement.element.setElement(node);
        if (restorePositionRef.current) {
            draggableElement.moveToCoordinates(restorePositionRef.current, node.getBoundingClientRect());
        }
        return () => {
            offDomEvents();
            offMove();
        };
    }, [draggableElement]));
    const boundingCallbackRef = useSafeRefCallback(useCallback((node) => {
        if (!node) {
            return;
        }
        return boundingElement.element.setElement(node);
    }, [boundingElement]));
    return [draggableCallbackRef, boundingCallbackRef, handleElementCallbackRef];
};
//# sourceMappingURL=DraggableCore.js.map