import { LlmSchemaComposer } from '@samchon/openapi/lib/composers/LlmSchemaComposer.mjs';
import ts from 'typescript';
import { MetadataCollection } from '../../factories/MetadataCollection.mjs';
import { MetadataFactory } from '../../factories/MetadataFactory.mjs';
import { TransformerError } from '../../transformers/TransformerError.mjs';

var LlmModelPredicator;
(function (LlmModelPredicator) {
    LlmModelPredicator.getConfig = (props) => {
        if (props.node === undefined)
            return undefined;
        const type = props.context.checker.getTypeFromTypeNode(props.node);
        const collection = new MetadataCollection();
        const result = MetadataFactory.analyze({
            checker: props.context.checker,
            transformer: props.context.transformer,
            options: {
                absorb: true,
                escape: false,
                constant: true,
                functional: false,
            },
            collection,
            type,
        });
        if (result.success === false)
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `Failed to analyze generic argument "Config".`,
            });
        const meta = result.data;
        if (meta.size() !== 1 ||
            meta.objects.length !== 1 ||
            meta.nullable === true ||
            meta.isRequired() === false)
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `Invalid generic argument "Config". It must be a literal object type.`,
            });
        const obj = meta.objects[0];
        if (obj.type.properties.some((p) => p.key.isSoleLiteral() === false))
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `Invalid generic argument "Config". It must be a literal object type. Do not allow dynamic properties.`,
            });
        else if (obj.type.properties.some((p) => p.value.size() !== 1 ||
            p.value.constants.length !== 1 ||
            p.value.nullable === true ||
            p.value.isRequired() === false))
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `Invalid generic argument "Config". It must be a literal object type. Do not allow variable type.`,
            });
        const config = {};
        for (const prop of obj.type.properties) {
            const key = prop.key.getSoleLiteral();
            const value = prop.value.constants[0].values[0].value;
            if (typeof value === "bigint")
                throw new TransformerError({
                    code: `typia.llm.${props.method}`,
                    message: `Invalid generic argument "Config". It must be a literal object type. Do not allow bigint.`,
                });
            config[key] = value;
        }
        return config;
    };
    LlmModelPredicator.getModel = (props) => {
        if (props.node === undefined)
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `generic argument "Model" must be specified.`,
            });
        // CHECK LITERAL TYPE
        const type = props.checker.getTypeFromTypeNode(props.node);
        if (!type.isLiteral() &&
            (type.getFlags() & ts.TypeFlags.BooleanLiteral) === 0)
            throw new TransformerError({
                code: `typia.llm.${props.method}`,
                message: `generic argument "Model" must be constant.`,
            });
        // GET VALUE AND VALIDATE IT
        const value = type.isLiteral()
            ? type.value
            : props.checker.typeToString(type);
        if (typeof value !== "string" ||
            LlmSchemaComposer.defaultConfig(value) === undefined)
            throw new TransformerError({
                code: "typia.llm.schema",
                message: `invalid value on generic argument "Model".`,
            });
        return value;
    };
})(LlmModelPredicator || (LlmModelPredicator = {}));

export { LlmModelPredicator };
//# sourceMappingURL=LlmModelPredicator.mjs.map
